pragma Singleton

import QtQuick 2.15

QtObject {
    id: transcriptionLanguages

    // Curated list of transcription languages for Whisper models.
    // With a searchable picker UI, this can be longer without hurting UX.
    readonly property var languages: [
        { code: "", name: qsTr("Automatic (detect)") },
        { code: "en", name: "English", autonym: "English" },
        { code: "zh", name: "Chinese", autonym: "中文" },
        { code: "es", name: "Spanish", autonym: "Español" },
        { code: "fr", name: "French", autonym: "Français" },
        { code: "de", name: "German", autonym: "Deutsch" },
        { code: "ja", name: "Japanese", autonym: "日本語" },
        { code: "pt", name: "Portuguese", autonym: "Português" },
        { code: "it", name: "Italian", autonym: "Italiano" },
        { code: "ko", name: "Korean", autonym: "한국어" },
        { code: "ru", name: "Russian", autonym: "Русский" },
        { code: "ar", name: "Arabic", autonym: "العربية" },
        { code: "nl", name: "Dutch", autonym: "Nederlands" },
        { code: "pl", name: "Polish", autonym: "Polski" },
        { code: "sv", name: "Swedish", autonym: "Svenska" },
        { code: "da", name: "Danish", autonym: "Dansk" },
        { code: "no", name: "Norwegian", autonym: "Norsk" },
        { code: "fi", name: "Finnish", autonym: "Suomi" },
        { code: "el", name: "Greek", autonym: "Ελληνικά" },
        { code: "cs", name: "Czech", autonym: "Čeština" },
        { code: "hu", name: "Hungarian", autonym: "Magyar" },
        { code: "ro", name: "Romanian", autonym: "Română" },
        { code: "uk", name: "Ukrainian", autonym: "Українська" },
        { code: "tr", name: "Turkish", autonym: "Türkçe" },
        { code: "hi", name: "Hindi", autonym: "हिन्दी" },
        { code: "bn", name: "Bengali", autonym: "বাংলা" },
        { code: "ur", name: "Urdu", autonym: "اردو" },
        { code: "te", name: "Telugu", autonym: "తెలుగు" },
        { code: "mr", name: "Marathi", autonym: "मराठी" },
        { code: "gu", name: "Gujarati", autonym: "ગુજરાતી" },
        { code: "pa", name: "Punjabi", autonym: "ਪੰਜਾਬੀ" },
        { code: "ml", name: "Malayalam", autonym: "മലയാളം" },
        { code: "kn", name: "Kannada", autonym: "ಕನ್ನಡ" },
        { code: "id", name: "Indonesian", autonym: "Bahasa Indonesia" },
        { code: "vi", name: "Vietnamese", autonym: "Tiếng Việt" },
        { code: "th", name: "Thai", autonym: "ไทย" },
        { code: "ms", name: "Malay", autonym: "Bahasa Melayu" },
        { code: "he", name: "Hebrew", autonym: "עברית" },
        { code: "ca", name: "Catalan", autonym: "Català" },
        { code: "sk", name: "Slovak", autonym: "Slovenčina" },
        { code: "bg", name: "Bulgarian", autonym: "Български" },
        { code: "hr", name: "Croatian", autonym: "Hrvatski" },
        { code: "sr", name: "Serbian", autonym: "Српски" },
        { code: "sl", name: "Slovenian", autonym: "Slovenščina" },
        { code: "lt", name: "Lithuanian", autonym: "Lietuvių" },
        { code: "lv", name: "Latvian", autonym: "Latviešu" },
        { code: "et", name: "Estonian", autonym: "Eesti" },
        { code: "fa", name: "Persian", autonym: "فارسی" },
        { code: "ta", name: "Tamil", autonym: "தமிழ்" },
        { code: "yue", name: "Cantonese", autonym: "粵語" },
        { code: "tl", name: "Tagalog", autonym: "Tagalog" },
        { code: "sw", name: "Swahili", autonym: "Kiswahili" },
        { code: "ne", name: "Nepali", autonym: "नेपाली" },
        { code: "my", name: "Burmese", autonym: "မြန်မာ" },
        { code: "km", name: "Khmer", autonym: "ភាសាខ្មែរ" },
        { code: "lo", name: "Lao", autonym: "ລາວ" },
        { code: "si", name: "Sinhala", autonym: "සිංහල" },
        { code: "af", name: "Afrikaans", autonym: "Afrikaans" },
        { code: "ps", name: "Pashto", autonym: "پښتو" },
        { code: "uz", name: "Uzbek", autonym: "Oʻzbek" },
        { code: "jw", name: "Javanese", autonym: "Basa Jawa" }
    ]

    readonly property var commonCodes: [
        "",
        "en",
        "es",
        "fr",
        "de",
        "pt",
        "it",
        "ru",
        "ar",
        "zh",
        "ja",
        "ko",
        "hi",
        "bn",
        "tr",
        "vi",
        "th",
        "id"
    ]

    // Helper to find index by language code
    function indexOfCode(code: string): int {
        for (var i = 0; i < languages.length; i++) {
            if (languages[i].code === code) {
                return i
            }
        }
        return 0  // Default to "Automatic"
    }

    function entryForCode(code: string): var {
        for (var i = 0; i < languages.length; i++) {
            if (languages[i].code === code) {
                return languages[i]
            }
        }
        return null
    }

    function displayName(entry: var, includeAutonym = true): string {
        if (!entry) return ""
        const baseName = entry.name || ""
        const autonym = entry.autonym || ""
        if (!includeAutonym || !autonym || autonym === baseName) {
            return baseName
        }
        return baseName + " — " + autonym
    }

    function displayNameForCode(code: string, includeAutonym = true): string {
        const entry = entryForCode(code)
        if (!entry) return code
        return displayName(entry, includeAutonym)
    }

    // RTL language codes
    readonly property var rtlCodes: ["ar", "he", "ur", "fa", "ps"]

    function isRTL(code: string): bool {
        return rtlCodes.indexOf(code) >= 0
    }
}
