pragma ComponentBehavior: Bound

import QtQuick 2.15
import QtQuick.Controls 2.15
import QtQuick.Layouts 1.15

import styles 1.0
import components.controls 1.0 as Controls

Popup {
    id: root
    modal: true
    Overlay.modal: Rectangle { color: Qt.rgba(0, 0, 0, 0.7) }
    focus: true
    padding: 0
    closePolicy: Popup.NoAutoClose  // Must click button to close

    required property var settingsModel
    required property var languageController

    // Local state for the switches (defaults)
    property bool checkUpdates: true
    property bool allowTelemetry: false
    property bool showHints: true

    width: Math.min(parent.width * 0.9, 500)
    height: Math.min(parent.height * 0.9, 600)

    background: Rectangle {
        radius: Theme.panelRadius * 2
        color: Theme.surface
        border.color: Theme.surfaceBorder
        border.width: 1
    }

    contentItem: Item {
        // Floating Discord button in top-right
        ToolButton {
            z: 1
            anchors.top: parent.top
            anchors.right: parent.right
            anchors.margins: Theme.spacingLg
            anchors.topMargin: 40
            implicitWidth: 40
            implicitHeight: 40
            icon.source: Theme.iconUrl("brand-discord")
            icon.width: 30
            icon.height: 30
            icon.color: Theme.textSecondary
            onClicked: Qt.openUrlExternally("https://discord.gg/uYGkyfzU4c")
            ToolTip.text: qsTr("Join our Discord")
            ToolTip.visible: hovered
        }

        ColumnLayout {
            anchors.fill: parent
            anchors.margins: Theme.spacingXl
            spacing: Theme.spacingLg

            // Scrollable content area
            Flickable {
                Layout.fillWidth: true
                Layout.fillHeight: true
                contentWidth: width
                contentHeight: scrollContent.implicitHeight
                clip: true
                boundsBehavior: Flickable.StopAtBounds

                ColumnLayout {
                    id: scrollContent
                    width: parent.width
                    spacing: Theme.spacingLg

                    // Header with app logo and welcome text
                    ColumnLayout {
                        Layout.fillWidth: true
                        spacing: Theme.spacingMd
                        Layout.alignment: Qt.AlignHCenter

                        Image {
                            source: Theme.imageUrl("logo_head.png")
                            sourceSize.width: 72
                            sourceSize.height: 72
                            Layout.alignment: Qt.AlignHCenter
                        }

                        Label {
                            text: qsTr("Welcome to Smart Media Cutter!")
                            font.pixelSize: 26
                            font.weight: Font.DemiBold
                            color: Theme.textPrimary
                            Layout.alignment: Qt.AlignHCenter
                        }

                        Label {
                            text: qsTr("Let's get you set up.")
                            font.pixelSize: 14
                            color: Theme.textSecondary
                            Layout.alignment: Qt.AlignHCenter
                            wrapMode: Text.WordWrap
                            horizontalAlignment: Text.AlignHCenter
                            Layout.fillWidth: true
                            lineHeight: 1.3
                        }
                    }

                    Item { Layout.preferredHeight: Theme.spacingSm }

                    // Language selection
                    RowLayout {
                        Layout.fillWidth: true
                        spacing: Theme.spacingMd

                        ColumnLayout {
                            Layout.fillWidth: true
                            spacing: 2

                            Label {
                                text: qsTr("Language")
                                font.pixelSize: 14
                                font.weight: Font.Medium
                                color: Theme.textPrimary
                            }

                            Label {
                                text: qsTr("Choose your preferred language")
                                font.pixelSize: 12
                                color: Theme.textSecondary
                                wrapMode: Text.WordWrap
                                Layout.fillWidth: true
                            }
                        }

                        ComboBox {
                            id: languageCombo
                            Layout.preferredWidth: 150
                            textRole: "name"
                            valueRole: "code"
                            model: root.languageController ? root.languageController.availableLanguages : []

                            Component.onCompleted: {
                                if (root.languageController) {
                                    var currentLang = root.languageController.currentLanguage
                                    for (var i = 0; i < model.length; i++) {
                                        if (model[i].code === currentLang) {
                                            currentIndex = i
                                            return
                                        }
                                    }
                                }
                            }

                            onActivated: function(index) {
                                if (root.languageController && model[index]) {
                                    root.languageController.setLanguage(model[index].code)
                                }
                            }
                        }
                    }

                    Rectangle {
                        Layout.fillWidth: true
                        implicitHeight: 1
                        color: Theme.surfaceBorder
                    }

                    // Settings switches
                    ColumnLayout {
                        Layout.fillWidth: true
                        spacing: Theme.spacingLg

                        // Update check toggle
                        RowLayout {
                            Layout.fillWidth: true
                            spacing: Theme.spacingMd

                            ColumnLayout {
                                Layout.fillWidth: true
                                spacing: 2

                                Label {
                                    text: qsTr("Check for updates on startup")
                                    font.pixelSize: 14
                                    font.weight: Font.Medium
                                    color: Theme.textPrimary
                                }

                                Label {
                                    text: qsTr("We'll let you know when a new version is available")
                                    font.pixelSize: 12
                                    color: Theme.textSecondary
                                    wrapMode: Text.WordWrap
                                    Layout.fillWidth: true
                                }
                            }

                            Switch {
                                id: updateSwitch
                                checked: root.checkUpdates
                                onToggled: root.checkUpdates = checked
                            }
                        }

                        Rectangle {
                            Layout.fillWidth: true
                            implicitHeight: 1
                            color: Theme.surfaceBorder
                        }


                        // Tutorial hints toggle
                        RowLayout {
                            Layout.fillWidth: true
                            spacing: Theme.spacingMd

                            ColumnLayout {
                                Layout.fillWidth: true
                                spacing: 2

                                Label {
                                    text: qsTr("Show tutorial hints")
                                    font.pixelSize: 14
                                    font.weight: Font.Medium
                                    color: Theme.textPrimary
                                }

                                Label {
                                    text: qsTr("Display helpful tips as you learn the app")
                                    font.pixelSize: 12
                                    color: Theme.textSecondary
                                    wrapMode: Text.WordWrap
                                    Layout.fillWidth: true
                                }
                            }

                            Switch {
                                id: hintsSwitch
                                checked: root.showHints
                                onToggled: root.showHints = checked
                            }
                        }

                        Rectangle {
                            Layout.fillWidth: true
                            implicitHeight: 1
                            color: Theme.surfaceBorder
                        }

                        // Telemetry toggle
                        RowLayout {
                            Layout.fillWidth: true
                            spacing: Theme.spacingMd

                            ColumnLayout {
                                Layout.fillWidth: true
                                spacing: 2

                                Label {
                                    text: qsTr("Allow collection of anonymous usage data")
                                    font.pixelSize: 14
                                    font.weight: Font.Medium
                                    color: Theme.textPrimary
                                }

                                Label {
                                    text: qsTr("Help us improve the app by sharing anonymous analytics")
                                    font.pixelSize: 12
                                    color: Theme.textSecondary
                                    wrapMode: Text.WordWrap
                                    Layout.fillWidth: true
                                }
                            }

                            Switch {
                                id: telemetrySwitch
                                checked: root.allowTelemetry
                                onToggled: root.allowTelemetry = checked
                            }
                        }

                    }
                }
            }

            // Fixed footer - always visible
            Controls.PrimaryButton {
                text: qsTr("Start Editing")
                Layout.fillWidth: true
                Layout.preferredHeight: 48
                font.pixelSize: 16

                onClicked: {
                    // Save settings
                    if (root.settingsModel) {
                        root.settingsModel.set_check_updates_on_startup(root.checkUpdates)
                        root.settingsModel.set_allow_telemetry(root.allowTelemetry)
                        root.settingsModel.set_hints_enabled(root.showHints)
                    }
                    root.close()
                }
            }

            Label {
                text: qsTr("You can change these anytime in Preferences.")
                font.pixelSize: 11
                color: Theme.textSecondary
                Layout.alignment: Qt.AlignHCenter
                Layout.bottomMargin: Theme.spacingSm
            }
        }
    }
}
