pragma ComponentBehavior: Bound

import QtQuick 2.15
import QtQuick.Controls 2.15
import QtQuick.Layouts 1.15

import styles 1.0
import components.controls 1.0 as Controls
import components.forms 1.0 as Forms
import components.layout 1.0

Item {
    id: root

    // Required properties
    required property var controller
    required property var licenseManager
    required property var languageController
    required property var toastManager
    required property var settingsModel

    // Signal to close panel
    signal closeRequested()

    // Cost estimation state
    property int estimatedCredits: 0

    function calculateCost() {
        if (!root.controller) return
        const segmentCount = root.controller.transcriptSegments.length
        // 1 credit per 5 segments, minimum 1 credit
        root.estimatedCredits = Math.max(1, Math.ceil(segmentCount / 8))
        if (root.licenseManager) {
            root.licenseManager.refreshCredits()
        }
    }

    // Reset cost estimate when panel becomes visible or language changes
    function resetEstimate() {
        root.estimatedCredits = 0
    }

    ColumnLayout {
        anchors.fill: parent
        spacing: 0

        // Header with back button
        SidebarHeader {
            title: qsTr("Translate Transcript")
            showBackButton: true
            onBackClicked: root.closeRequested()
        }

        // Content scroll area
        ScrollView {
            Layout.fillWidth: true
            Layout.fillHeight: true
            clip: true
            contentWidth: availableWidth

            ColumnLayout {
                width: parent.width
                spacing: Theme.spacingLg

                Item {
                    Layout.fillWidth: true
                    Layout.preferredHeight: contentColumn.height

                    ColumnLayout {
                        id: contentColumn
                        anchors.left: parent.left
                        anchors.right: parent.right
                        anchors.top: parent.top
                        anchors.margins: Theme.spacingLg
                        spacing: Theme.spacingMd

                        // Target language selection
                        Label {
                            text: qsTr("Translate to")
                            font.pixelSize: 12
                            color: Theme.textSecondary
                        }

                        Forms.LanguagePickerField {
                            id: targetLanguagePicker
                            Layout.fillWidth: true
                            showAutomatic: false
                            code: root.settingsModel ? root.settingsModel.translation_target_language() : "en"
                            settingsModel: root.settingsModel
                            enabled: root.controller && root.controller.translationStatus !== "translating"
                            onCodeSelected: (code) => {
                                root.resetEstimate()
                                if (root.settingsModel && code) {
                                    root.settingsModel.set_translation_target_language(code)
                                }
                            }
                        }

                        // Cost estimation display
                        ColumnLayout {
                            Layout.fillWidth: true
                            spacing: 4
                            visible: root.estimatedCredits > 0

                            Label {
                                property int credits: root.estimatedCredits
                                property string dollars: (credits / 4000).toFixed(2)
                                text: qsTr("Estimated cost: %1 credits ($%2)").arg(credits).arg(dollars)
                                color: Theme.textPrimary
                                font.pixelSize: 13
                                font.weight: Font.DemiBold
                                wrapMode: Text.WordWrap
                                Layout.fillWidth: true
                            }

                            Label {
                                text: qsTr("Final cost may vary based on text complexity and target language.")
                                color: Theme.textSecondary
                                font.pixelSize: 12
                                wrapMode: Text.WordWrap
                                Layout.fillWidth: true
                            }

                            // Percentage of remaining credits
                            Label {
                                property int remaining: root.licenseManager ?
                                    (root.licenseManager.creditsLimit - root.licenseManager.creditsUsed) : 0
                                property int estimated: root.estimatedCredits
                                property real percentage: remaining > 0 ? (estimated / remaining * 100) : 0
                                visible: root.licenseManager &&
                                         !root.licenseManager.creditsLoading &&
                                         remaining > 0
                                text: qsTr("This is %1% of your remaining credits").arg(percentage.toFixed(1))
                                color: Theme.textSecondary
                                font.pixelSize: 12
                                wrapMode: Text.WordWrap
                                Layout.fillWidth: true
                            }

                            // Warning: insufficient credits
                            Rectangle {
                                property int remaining: root.licenseManager ?
                                    (root.licenseManager.creditsLimit - root.licenseManager.creditsUsed) : 0
                                visible: root.licenseManager &&
                                         !root.licenseManager.creditsLoading &&
                                         root.estimatedCredits > remaining
                                Layout.fillWidth: true
                                Layout.topMargin: Theme.spacingSm
                                radius: Theme.panelRadius - 4
                                color: Qt.rgba(Theme.statusDanger.r, Theme.statusDanger.g, Theme.statusDanger.b, 0.15)
                                border.color: Qt.rgba(Theme.statusDanger.r, Theme.statusDanger.g, Theme.statusDanger.b, 0.4)
                                border.width: 1
                                implicitHeight: insufficientWarningLabel.implicitHeight + Theme.spacingSm * 2

                                Label {
                                    id: insufficientWarningLabel
                                    anchors.fill: parent
                                    anchors.margins: Theme.spacingSm
                                    text: qsTr("Insufficient credits for this translation.")
                                    color: Theme.statusDanger
                                    font.pixelSize: 12
                                    font.weight: Font.DemiBold
                                    wrapMode: Text.WordWrap
                                }
                            }
                        }

                        // Calculate cost button
                        Button {
                            Layout.fillWidth: true
                            text: qsTr("Calculate translation cost")
                            visible: root.estimatedCredits === 0
                            font.pixelSize: 12
                            onClicked: root.calculateCost()
                        }

                        // Translate button
                        Controls.PrimaryButton {
                            Layout.fillWidth: true
                            text: root.controller && root.controller.translationStatus === "translating"
                                ? qsTr("Translating...")
                                : qsTr("Translate")
                            enabled: root.controller
                                && root.controller.translationStatus !== "translating"
                                && targetLanguagePicker.code !== ""
                            onClicked: {
                                if (root.controller) {
                                    root.controller.start_translation(targetLanguagePicker.code)
                                }
                            }
                        }

                        // Progress bar
                        ProgressBar {
                            Layout.fillWidth: true
                            from: 0; to: 100
                            value: root.controller ? root.controller.translationProgress : 0
                            visible: root.controller && root.controller.translationStatus === "translating"
                        }

                        Rectangle {
                            Layout.fillWidth: true
                            Layout.preferredHeight: 1
                            color: Theme.surfaceBorder
                            visible: viewSection.visible
                        }

                        // View selection section
                        ColumnLayout {
                            id: viewSection
                            Layout.fillWidth: true
                            spacing: Theme.spacingSm
                            visible: root.controller && root.controller.availableTranslations.length > 0

                            Label {
                                text: qsTr("View")
                                font.pixelSize: 12
                                color: Theme.textSecondary
                            }

                            RadioButton {
                                text: qsTr("Original")
                                checked: root.controller && root.controller.currentTranslationView === ""
                                onClicked: {
                                    if (root.controller) {
                                        root.controller.set_translation_view("")
                                    }
                                }
                            }

                            Repeater {
                                model: root.controller ? root.controller.availableTranslations : []

                                RadioButton {
                                    required property string modelData
                                    text: TranscriptionLanguages.displayNameForCode(modelData, true)
                                    checked: root.controller && root.controller.currentTranslationView === modelData
                                    onClicked: {
                                        if (root.controller) {
                                            root.controller.set_translation_view(modelData)
                                        }
                                    }
                                }
                            }
                        }
                    }
                }

                Item { Layout.fillHeight: true }

                // Info text at bottom
                Label {
                    Layout.fillWidth: true
                    Layout.margins: Theme.spacingLg
                    text: qsTr("Translation uses cloud AI.")
                    font.pixelSize: 11
                    color: Theme.textSecondary
                    opacity: 0.7
                    wrapMode: Text.WordWrap
                    horizontalAlignment: Text.AlignHCenter
                }
            }
        }
    }
}
