import QtQuick 2.15
import QtQuick.Controls 2.15
import QtQuick.Layouts 1.15
import QtQuick.Window 2.15
import styles 1.0

Popup {
    id: popover
    property var controller
    property var appController
    property bool selectionActive: false
    property bool selectionInView: true
    property rect selectionRect: Qt.rect(0, 0, 0, 0)
    property real selectionTimeStart: 0
    property real selectionTimeEnd: 0
    property int selectionSegmentId: 0
    property Item sourceItem: null

    signal dismissed()

    // Check if the popover's bounding box is within the sourceItem bounds
    readonly property bool popoverInBounds: {
        if (!sourceItem || !parent || !visible) return true
        const sourceRect = sourceItem.mapToItem(parent, 0, 0, sourceItem.width, sourceItem.height)
        return y >= sourceRect.y &&
               y + height <= sourceRect.y + sourceRect.height &&
               x >= sourceRect.x &&
               x + width <= sourceRect.x + sourceRect.width
    }

    // Hide when popover would be drawn outside the transcript area
    opacity: selectionInView && popoverInBounds ? 1.0 : 0.0
    Behavior on opacity { NumberAnimation { duration: 100 } }

    // Explicitly parent to the overlay for predictable positioning
    parent: Overlay.overlay

    modal: false
    closePolicy: Popup.NoAutoClose
    padding: Theme.spacingSm

    // X: clamp within sourceItem bounds (stays within transcript area)
    // Y: escape to overlay coordinates (can go above transcript area)
    x: {
        if (!sourceItem || !parent) return 0
        const sourceLeft = sourceItem.mapToItem(parent, 0, 0).x
        const localX = Math.max(Theme.spacingSm,
                           Math.min(selectionRect.x + selectionRect.width / 2 - width / 2,
                                    sourceItem.width - width - Theme.spacingSm))
        return sourceLeft + localX
    }
    y: {
        if (!sourceItem || !parent) return 0
        const globalPos = sourceItem.mapToItem(parent, 0, selectionRect.y)
        return globalPos.y - height - Theme.spacingSm
    }

    background: Rectangle {
        color: Theme.surfaceStrong
        radius: Theme.panelRadius
        border.color: Theme.surfaceBorder
    }

    contentItem: RowLayout {
        spacing: Theme.spacingSm
        anchors.margins: Theme.spacingSm
        anchors.fill: parent

        Button {
            text: qsTr("Play")
            onClicked: {
                if (popover.appController) {
                    popover.appController.seekToTime(popover.selectionTimeStart)
                    if (popover.appController.play)
                        popover.appController.play()
                }
            }
        }

        Button {
            readonly property bool hasMatchingLoop: (function() {
                try {
                    var lr = popover.appController ? popover.appController.loopRange : []
                    if (!lr || lr.length !== 2)
                        return false
                    if (!popover.appController || !popover.appController.loopEnabled)
                        return false
                    var a = Math.min(popover.selectionTimeStart, popover.selectionTimeEnd)
                    var b = Math.max(popover.selectionTimeStart, popover.selectionTimeEnd)
                    return Math.abs(lr[0] - a) < 0.01 && Math.abs(lr[1] - b) < 0.01
                } catch (e) {
                    return false
                }
            })()
            text: hasMatchingLoop ? qsTr("Clear Loop") : qsTr("Loop")
            onClicked: {
                if (!popover.appController)
                    return
                var a = Math.min(popover.selectionTimeStart, popover.selectionTimeEnd)
                var b = Math.max(popover.selectionTimeStart, popover.selectionTimeEnd)
                if (this.hasMatchingLoop) {
                    popover.appController.clearLoop()
                } else {
                    popover.appController.loopSelectionRange(a, b)
                }
            }
        }

        Button {
            text: qsTr("Cut")
            objectName: "popoverCutButton"
            onClicked: {
                if (popover.appController) {
                    var a = Math.min(popover.selectionTimeStart, popover.selectionTimeEnd)
                    var b = Math.max(popover.selectionTimeStart, popover.selectionTimeEnd)
                    popover.appController.addCutRange(a, b)
                }
            }
        }

        Button {
            text: qsTr("Uncut")
            onClicked: {
                if (popover.appController) {
                    var a = Math.min(popover.selectionTimeStart, popover.selectionTimeEnd)
                    var b = Math.max(popover.selectionTimeStart, popover.selectionTimeEnd)
                    popover.appController.removeCutRange(a, b)
                }
            }
        }

        // Close button
        ToolButton {
            id: closeBtn
            implicitWidth: 28
            implicitHeight: 28
            icon.source: Theme.iconUrl("x")
            icon.width: 14
            icon.height: 14
            icon.color: Theme.textSecondary
            onClicked: popover.dismissed()

            background: Rectangle {
                radius: 4
                color: closeBtn.pressed
                    ? Qt.rgba(Theme.textPrimary.r, Theme.textPrimary.g, Theme.textPrimary.b, 0.12)
                    : closeBtn.hovered
                        ? Qt.rgba(Theme.textPrimary.r, Theme.textPrimary.g, Theme.textPrimary.b, 0.06)
                        : "transparent"
            }
        }
    }
}
