import QtQuick 2.15
import QtQuick.Controls 2.15
import QtQuick.Layouts 1.15
import styles 1.0

FocusScope {
    id: root
    objectName: "shortcutEditor"

    Layout.fillWidth: true

    // qmllint disable missing-property
    property var settingsModel
    // qmllint enable missing-property

    property alias label: caption.text
    property string shortcutId: ""
    property var sequences: []  // List of shortcut sequences
    property bool recording: false

    implicitHeight: layout.implicitHeight
    implicitWidth: layout.implicitWidth
    focus: recording

    function startRecording() {
        recording = true
        forceActiveFocus()
    }

    function addSequence(value) {
        if (!value || value.length === 0) return
        // Check if already exists
        for (var i = 0; i < sequences.length; i++) {
            if (sequences[i] === value) {
                recording = false
                return
            }
        }
        // Add new shortcut
        if (shortcutId.length > 0) {
            var newSeqs = settingsModel.add_shortcut(shortcutId, value)
            sequences = newSeqs
        }
        recording = false
    }

    function removeSequence(seq) {
        if (shortcutId.length > 0) {
            var newSeqs = settingsModel.remove_shortcut(shortcutId, seq)
            sequences = newSeqs
        }
    }

    Keys.onPressed: function(event) {
        if (!recording) {
            return
        }

        if (event.key === Qt.Key_Escape) {
            event.accepted = true
            recording = false
            return
        }

        var seq = settingsModel.encode_shortcut(event.key, event.modifiers)
        if (seq.length === 0) {
            event.accepted = true
            return
        }

        event.accepted = true
        addSequence(seq)
    }

    RowLayout {
        id: layout
        spacing: Theme.spacingMd

        Label {
            id: caption
            Layout.preferredWidth: 200 // Fixed width for alignment
            Layout.alignment: Qt.AlignVCenter
            text: "Shortcut"
            font.weight: Font.Medium
            color: Theme.textPrimary
            elide: Text.ElideRight
        }

        // Container for shortcut chips
        Frame {
            Layout.preferredWidth: 240
            Layout.fillWidth: false
            padding: 6
            background: Rectangle {
                radius: 6
                color: Theme.surface
                border.color: root.recording ? Theme.accent : Theme.surfaceBorder
                border.width: 1
            }

            contentItem: Flow {
                id: chipsFlow
                spacing: 4
                width: parent.width

                // Shortcut chips
                Repeater {
                    model: root.sequences

                    Rectangle {
                        id: chip
                        required property string modelData
                        property var editor: root  // qmllint disable unqualified
                        height: 24
                        width: chipRow.width + 8
                        radius: 4
                        color: Theme.surfaceRaised
                        border.color: Theme.surfaceBorder
                        border.width: 1

                        RowLayout {
                            id: chipRow
                            anchors.centerIn: parent
                            spacing: 2

                            Label {
                                text: chip.modelData
                                font.pixelSize: 12
                                font.family: "Monospace"
                                color: Theme.textPrimary
                            }

                            // Remove button
                            Rectangle {
                                Layout.preferredWidth: 14
                                Layout.preferredHeight: 14
                                radius: 7
                                color: removeArea.containsMouse ? Theme.statusDanger : "transparent"

                                Label {
                                    anchors.centerIn: parent
                                    text: "×"
                                    font.pixelSize: 12
                                    font.weight: Font.Bold
                                    color: removeArea.containsMouse ? Theme.textOnAccent : Theme.textSecondary
                                }

                                MouseArea {
                                    id: removeArea
                                    anchors.fill: parent
                                    hoverEnabled: true
                                    cursorShape: Qt.PointingHandCursor
                                    onClicked: chip.editor.removeSequence(chip.modelData)
                                }
                            }
                        }
                    }
                }

                // Empty state or recording indicator
                Label {
                    visible: root.sequences.length === 0 && !root.recording
                    text: qsTr("Not set")
                    font.pixelSize: 12
                    font.italic: true
                    color: Theme.textSecondary
                    height: 24
                    verticalAlignment: Text.AlignVCenter
                }

                Label {
                    visible: root.recording
                    text: qsTr("Press keys...")
                    font.pixelSize: 12
                    font.italic: true
                    color: Theme.accent
                    height: 24
                    verticalAlignment: Text.AlignVCenter
                }
            }
        }

        Button {
            text: root.recording ? qsTr("Cancel") : qsTr("+ Add")
            flat: false
            Layout.preferredWidth: 80
            onClicked: {
                if (root.recording) {
                    root.recording = false
                } else {
                    root.startRecording()
                }
            }
        }

        Item { Layout.fillWidth: true } // Spacer
    }

    Component.onCompleted: {
        if (shortcutId.length > 0) {
            sequences = settingsModel.shortcuts(shortcutId)
        }
    }

    Connections {
        target: root.settingsModel
        function onShortcut_changed(name, seqs) {
            if (name === root.shortcutId && !root.recording) {
                root.sequences = seqs
            }
        }
    }
}
