pragma ComponentBehavior: Bound

import QtQuick 2.15
import QtQuick.Controls 2.15
import QtQuick.Layouts 1.15
import QtQuick.Dialogs

import styles 1.0
import components.controls 1.0 as Controls
import components.notifications 1.0 as Notifications
import "../../../utils" as Utils

Popup {
    id: root
    modal: true
    Overlay.modal: Rectangle { color: Qt.rgba(0, 0, 0, 0.6) }
    focus: true
    padding: 0
    closePolicy: Popup.CloseOnEscape

    required property var recoveryController
    required property var licenseManager
    required property var languageController
    required property var toastManager
    property var appWorkspaceController: null  // For loading recovered file

    // Computed from controller
    readonly property string corruptPath: recoveryController ? recoveryController.corruptPath : ""
    readonly property string referencePath: recoveryController ? recoveryController.referencePath : ""
    readonly property string outputPath: recoveryController ? recoveryController.outputPath : ""
    readonly property string status: recoveryController ? recoveryController.status : "idle"
    readonly property string errorMessage: recoveryController ? recoveryController.errorMessage : ""
    readonly property bool canRecover: recoveryController ? recoveryController.canRecover : false
    readonly property int progressPercent: recoveryController ? recoveryController.progressPercent : 0
    readonly property string progressMessage: recoveryController ? recoveryController.progressMessage : ""
    readonly property int referenceAudioTrackCount: recoveryController ? recoveryController.referenceAudioTrackCount : 0
    readonly property bool hasMultipleAudioTracks: recoveryController ? recoveryController.hasMultipleAudioTracks : false
    readonly property bool hasLicense: licenseManager ? licenseManager.isValid() : false

    // Signals
    signal loadRecoveredFile(string path)

    width: Math.min(parent ? parent.width * 0.9 : 700, 720)
    height: Math.min(parent ? parent.height * 0.9 : 580, 700)

    background: Rectangle {
        radius: Theme.panelRadius * 2
        color: Theme.surface
        border.color: Theme.surfaceBorder
        border.width: 1
    }

    onClosed: {
        if (root.recoveryController) {
            root.recoveryController.reset()
        }
    }

    contentItem: ColumnLayout {
        anchors.fill: parent
        anchors.margins: Theme.spacingXl
        spacing: Theme.spacingLg

        // Header with title and close button
        RowLayout {
            Layout.fillWidth: true

            Label {
                text: qsTr("Recover Corrupt Video")
                font.pixelSize: 22
                font.weight: Font.DemiBold
                color: Theme.textPrimary
                Layout.fillWidth: true
            }

            ToolButton {
                Accessible.name: qsTr("Close")
                text: "\u2715"
                font.pixelSize: 18
                implicitWidth: 36
                implicitHeight: 36
                onClicked: root.close()
            }
        }

        // Info box
        Rectangle {
            Layout.fillWidth: true
            color: Theme._withAlpha(Theme.accent, 0.08)
            radius: Theme.panelRadius
            implicitHeight: infoLayout.implicitHeight + Theme.spacingMd * 2

            RowLayout {
                id: infoLayout
                anchors.fill: parent
                anchors.margins: Theme.spacingMd
                spacing: Theme.spacingSm

                Image {
                    source: Theme.iconUrl("alert-triangle")
                    sourceSize.width: 20
                    sourceSize.height: 20
                    Layout.alignment: Qt.AlignTop
                }

                Label {
                    Layout.fillWidth: true
                    wrapMode: Text.WordWrap
                    text: qsTr("This video file appears to be corrupt or truncated (missing moov atom). To recover it, provide a reference file recorded with the same camera or encoder settings.")
                    color: Theme.textSecondary
                    font.pixelSize: 13
                    lineHeight: 1.3
                }
            }
        }

        // Two column layout for file drop boxes
        RowLayout {
            Layout.fillWidth: true
            Layout.fillHeight: true
            Layout.minimumHeight: 180
            spacing: Theme.spacingLg

            // Corrupt file box (left, pre-filled, not interactive)
            FileDropBox {
                id: corruptDropBox
                Layout.fillWidth: true
                Layout.fillHeight: true
                label: qsTr("Corrupt File")
                filePath: root.corruptPath
                isCorrupt: true
                interactive: false
            }

            // Reference file box (right, user drops here)
            FileDropBox {
                id: referenceDropBox
                Layout.fillWidth: true
                Layout.fillHeight: true
                label: qsTr("Reference File")
                filePath: root.referencePath
                placeholder: qsTr("Drop a working video from the same source")
                interactive: root.status === "idle" || root.status === "error"

                onFileDropped: function(path) {
                    if (root.recoveryController) {
                        root.recoveryController.set_reference_path(path)
                    }
                }
                onBrowseClicked: referenceFileDialog.open()
            }
        }

        // Tip box
        Rectangle {
            Layout.fillWidth: true
            color: Theme.surfaceRaised
            radius: Theme.panelRadius - 2
            implicitHeight: tipLayout.implicitHeight + Theme.spacingSm * 2

            RowLayout {
                id: tipLayout
                anchors.fill: parent
                anchors.margins: Theme.spacingSm
                spacing: Theme.spacingSm

                Label {
                    text: "\u2139"  // Info symbol
                    color: Theme.textSecondary
                    font.pixelSize: 14
                    font.weight: Font.Bold
                    Layout.alignment: Qt.AlignTop
                }

                Label {
                    Layout.fillWidth: true
                    text: qsTr("The reference file must use the same codec, resolution, frame rate, and audio tracks as the corrupt file. A file from the same recording session works best.")
                    wrapMode: Text.WordWrap
                    color: Theme.textSecondary
                    font.pixelSize: 12
                    lineHeight: 1.2
                }
            }
        }

        // Output path field
        ColumnLayout {
            Layout.fillWidth: true
            spacing: Theme.spacingXs

            Label {
                text: qsTr("Output file")
                font.pixelSize: 13
                color: Theme.textSecondary
            }

            RowLayout {
                Layout.fillWidth: true
                spacing: Theme.spacingSm

                TextField {
                    id: outputPathField
                    Layout.fillWidth: true
                    text: root.outputPath
                    enabled: root.status === "idle" || root.status === "error"
                    selectByMouse: true
                    font.pixelSize: 13

                    onTextEdited: {
                        if (root.recoveryController) {
                            root.recoveryController.set_output_path(text)
                        }
                    }

                    background: Rectangle {
                        radius: Theme.panelRadius - 2
                        color: Theme.background
                        border.width: 1
                        border.color: outputPathField.activeFocus ? Theme.accent : Theme.surfaceBorder
                    }
                }

                Button {
                    id: browseOutputButton
                    text: qsTr("Browse")
                    enabled: root.status === "idle" || root.status === "error"
                    implicitHeight: outputPathField.implicitHeight
                    onClicked: outputFileDialog.open()

                    background: Rectangle {
                        radius: Theme.panelRadius - 2
                        color: browseOutputButton.down ? Theme.surfaceStrong : (browseOutputButton.hovered ? Theme.surfaceRaised : Theme.surface)
                        border.width: 1
                        border.color: Theme.surfaceBorder
                    }
                }
            }
        }

        // Audio tracks info (visible when reference has multiple tracks)
        Rectangle {
            Layout.fillWidth: true
            visible: root.hasMultipleAudioTracks && root.referencePath !== ""
            color: root.hasLicense ? Theme._withAlpha("#4CAF50", 0.08) : Theme._withAlpha(Theme.accent, 0.08)
            radius: Theme.panelRadius - 2
            implicitHeight: audioTracksRow.implicitHeight + Theme.spacingSm * 2

            RowLayout {
                id: audioTracksRow
                anchors.fill: parent
                anchors.margins: Theme.spacingSm
                spacing: Theme.spacingSm

                Image {
                    source: root.hasLicense ? Theme.iconUrl("check") : Theme.iconUrl("lock")
                    sourceSize.width: 16
                    sourceSize.height: 16
                    Layout.alignment: Qt.AlignVCenter
                }

                Label {
                    text: root.hasLicense
                        ? qsTr("Recovering %1 audio tracks").arg(root.referenceAudioTrackCount)
                        : qsTr("Recovering 1 of %1 audio tracks").arg(root.referenceAudioTrackCount)
                    color: Theme.textPrimary
                    font.pixelSize: 13
                    Layout.fillWidth: true
                }

                // Upgrade link (only shown without license)
                Label {
                    visible: !root.hasLicense
                    text: qsTr("Upgrade to recover all")
                    color: Theme.accent
                    font.pixelSize: 12
                    font.underline: true

                    MouseArea {
                        anchors.fill: parent
                        cursorShape: Qt.PointingHandCursor
                        onClicked: premiumPopup.open()
                    }
                }
            }
        }

        // Status area
        Rectangle {
            Layout.fillWidth: true
            Layout.maximumHeight: 120
            visible: root.status !== "idle"
            color: root.status === "error" ? Theme._withAlpha(Theme.statusDanger, 0.1)
                 : root.status === "completed" ? Theme._withAlpha("#4CAF50", 0.1)
                 : "transparent"
            radius: Theme.panelRadius - 2
            implicitHeight: statusColumn.implicitHeight + Theme.spacingSm * 2

            ColumnLayout {
                id: statusColumn
                anchors.fill: parent
                anchors.margins: Theme.spacingSm
                spacing: Theme.spacingXs

                // Progress indicator section
                ColumnLayout {
                    visible: root.status === "recovering"
                    Layout.fillWidth: true
                    spacing: Theme.spacingXs

                    // Status message
                    Label {
                        text: root.progressMessage || qsTr("Starting recovery...")
                        color: Theme.textPrimary
                        font.pixelSize: 13
                    }

                    // Progress bar
                    ProgressBar {
                        Layout.fillWidth: true
                        from: 0
                        to: 100
                        value: root.progressPercent
                        implicitHeight: 6

                        background: Rectangle {
                            radius: 3
                            color: Theme.surfaceStrong
                        }

                        contentItem: Item {
                            Rectangle {
                                width: parent.width * (root.progressPercent / 100)
                                height: parent.height
                                radius: 3
                                color: Theme.accent
                            }
                        }
                    }

                    // Percentage label
                    Label {
                        text: root.progressPercent + "%"
                        color: Theme.textSecondary
                        font.pixelSize: 11
                    }
                }

                // Success message
                Label {
                    visible: root.status === "completed"
                    text: qsTr("Recovery complete!")
                    color: "#4CAF50"
                    font.pixelSize: 13
                }

                // Error area with scrollable details
                ColumnLayout {
                    visible: root.status === "error"
                    Layout.fillWidth: true
                    Layout.fillHeight: true
                    spacing: Theme.spacingXs

                    // Error title
                    Label {
                        text: qsTr("Recovery failed")
                        color: Theme.statusDanger
                        font.pixelSize: 13
                        font.weight: Font.DemiBold
                    }

                    // Scrollable error details
                    ScrollView {
                        Layout.fillWidth: true
                        Layout.fillHeight: true
                        Layout.minimumHeight: 40
                        clip: true

                        TextArea {
                            id: errorTextArea
                            readOnly: true
                            selectByMouse: true
                            wrapMode: TextEdit.Wrap
                            font.pixelSize: 11
                            font.family: "monospace"
                            color: Theme.textSecondary
                            text: "Input: " + root.corruptPath + "\n" +
                                  "Reference: " + root.referencePath + "\n" +
                                  "Output: " + root.outputPath + "\n\n" +
                                  (root.errorMessage || "Unknown error")
                            background: null
                            padding: 0
                        }
                    }
                }
            }
        }

        // Action buttons
        RowLayout {
            Layout.fillWidth: true
            spacing: Theme.spacingSm

            Item { Layout.fillWidth: true }

            // Cancel/Close button
            Controls.SecondaryButton {
                text: root.status === "completed" ? qsTr("Close") : qsTr("Cancel")
                onClicked: {
                    if (root.status === "recovering" && root.recoveryController) {
                        root.recoveryController.cancel()
                    }
                    root.close()
                }
            }

            // Primary action button
            Controls.PrimaryButton {
                visible: root.status !== "completed"
                text: qsTr("Recover")
                enabled: root.canRecover && root.status !== "recovering"
                onClicked: {
                    if (root.recoveryController) {
                        root.recoveryController.start_recovery()
                    }
                }
            }

            // Open recovered file button (shown after success)
            Controls.PrimaryButton {
                visible: root.status === "completed"
                text: qsTr("Open Recovered File")
                onClicked: {
                    root.loadRecoveredFile(root.outputPath)
                    root.close()
                }
            }
        }
    }

    // File dialogs
    FileDialog {
        id: referenceFileDialog
        title: qsTr("Select Reference File")
        nameFilters: [qsTr("Video files") + " (*.mp4 *.mov *.m4v *.mkv)", qsTr("All files") + " (*)"]
        onAccepted: {
            var path = Utils.PathUtils.urlToPath(selectedFile)
            if (root.recoveryController) {
                root.recoveryController.set_reference_path(path)
            }
        }
    }

    FileDialog {
        id: outputFileDialog
        title: qsTr("Save Recovered File")
        fileMode: FileDialog.SaveFile
        nameFilters: [qsTr("MP4 files") + " (*.mp4)", qsTr("All files") + " (*)"]
        onAccepted: {
            var path = Utils.PathUtils.urlToPath(selectedFile)
            if (root.recoveryController) {
                root.recoveryController.set_output_path(path)
            }
        }
    }

    // Premium feature popup for multi-track recovery
    Notifications.PremiumFeaturePopup {
        id: premiumPopup
        parent: Overlay.overlay
        anchors.centerIn: parent
        featureName: qsTr("Multi-Track Audio Recovery")
        featureDescription: qsTr("Recover all audio tracks from your corrupt video file, not just the first one.")
        licenseManager: root.licenseManager
        languageController: root.languageController
        featureBenefits: [
            qsTr("Recover all audio tracks from corrupt multi-track recordings"),
            qsTr("Supports AAC and Opus audio tracks")
        ]
        onActivationSucceeded: root.toastManager.success(qsTr("License activated successfully!"), "")
    }
}
