pragma ComponentBehavior: Bound

import QtQuick 2.15
import QtQuick.Controls 2.15
import QtQuick.Layouts 1.15

import styles 1.0
import components.controls 1.0 as Controls
import components.forms 1.0 as Forms

Popup {
    id: root
    modal: true
    Overlay.modal: Rectangle { color: Qt.rgba(0, 0, 0, 0.6) }
    focus: true
    padding: 0
    closePolicy: Popup.CloseOnEscape | Popup.CloseOnPressOutside

    required property var feedbackService
    required property var licenseManager

    property bool hasValidLicense: false

    // Form state
    property int topicIndex: 0
    property string email: ""
    property string content: ""
    property bool sending: false
    property string errorMessage: ""

    readonly property var topics: [
        { value: "content_report", label: qsTr("Report Content Issue") },
        { value: "bug_report", label: qsTr("Bug Report") },
        { value: "feature_request", label: qsTr("Feature Request") },
        { value: "general", label: qsTr("General Feedback") }
    ]

    readonly property string selectedTopic: topics[topicIndex].value
    readonly property bool canSubmit: content.trim().length > 0 && !sending

    width: Math.min(parent ? parent.width * 0.9 : 500, 500)
    height: Math.min(parent ? parent.height * 0.9 : 500, 500)
    anchors.centerIn: parent

    background: Rectangle {
        radius: Theme.panelRadius
        color: Theme.surface
        border.color: Theme.surfaceBorder
        border.width: 1
    }

    onOpened: {
        root.topicIndex = 0
        root.email = ""
        root.content = ""
        root.errorMessage = ""
        root.sending = false
        root.hasValidLicense = root.licenseManager && root.licenseManager.isValid()
        contentArea.forceActiveFocus()
    }

    Connections {
        target: root.licenseManager
        function onLicense_status_changed(isValid: bool) {
            root.hasValidLicense = isValid
        }
    }

    Connections {
        target: root.feedbackService
        function onFeedbackSent() {
            root.sending = false
            root.close()
        }
        function onFeedbackFailed(error: string) {
            root.sending = false
            root.errorMessage = qsTr("Failed to send feedback. Please try again.")
        }
    }

    contentItem: Item {
        ColumnLayout {
            anchors.fill: parent
            anchors.margins: Theme.spacingLg
            spacing: Theme.spacingMd

            // Header
            RowLayout {
                Layout.fillWidth: true

                Label {
                    text: qsTr("Send Feedback")
                    font.pixelSize: 20
                    font.weight: Font.DemiBold
                    color: Theme.textPrimary
                    Layout.fillWidth: true
                }

                ToolButton {
                    icon.source: Theme.iconUrl("brand-discord")
                    icon.width: 20
                    icon.height: 20
                    icon.color: Theme.textSecondary
                    onClicked: Qt.openUrlExternally("https://discord.gg/uYGkyfzU4c")
                    ToolTip.text: qsTr("Join our Discord for direct support")
                    ToolTip.visible: hovered
                }

                ToolButton {
                    icon.source: Theme.iconUrl("x")
                    icon.width: 20
                    icon.height: 20
                    icon.color: Theme.textSecondary
                    onClicked: root.close()
                }
            }

        // Topic selector
        Forms.LabeledField {
            label: qsTr("Topic")
            Layout.fillWidth: true

            ComboBox {
                id: topicCombo
                Layout.fillWidth: true
                model: root.topics
                textRole: "label"
                currentIndex: root.topicIndex
                onCurrentIndexChanged: root.topicIndex = currentIndex
            }
        }

        // Email (optional)
        Forms.LabeledField {
            label: qsTr("Email")
            helperText: root.hasValidLicense
                ? qsTr("Optional - only needed if you want a response to a different email than your license")
                : qsTr("Optional - if you'd like a response")
            Layout.fillWidth: true

            TextField {
                id: emailField
                Layout.fillWidth: true
                placeholderText: qsTr("your@email.com")
                text: root.email
                onTextChanged: root.email = text
            }
        }

        // Active license indicator
        Label {
            visible: root.hasValidLicense
            text: qsTr("Your feedback will be linked to your active license")
            color: Theme.accent
            font.pixelSize: 12
            Layout.fillWidth: true
        }

        // Content
        Forms.LabeledField {
            label: qsTr("Message")
            required: true
            Layout.fillWidth: true
            Layout.fillHeight: true

            ScrollView {
                Layout.fillWidth: true
                Layout.fillHeight: true

                TextArea {
                    id: contentArea
                    placeholderText: qsTr("Describe your feedback...")
                    text: root.content
                    onTextChanged: root.content = text
                    wrapMode: TextArea.Wrap
                }
            }
        }

        // Error message
        Label {
            visible: root.errorMessage.length > 0
            text: root.errorMessage
            color: Theme.statusDanger
            font.pixelSize: 12
            Layout.fillWidth: true
            wrapMode: Label.Wrap
        }

        // Submit button
        RowLayout {
            Layout.fillWidth: true
            spacing: Theme.spacingMd

            Item { Layout.fillWidth: true }

            Controls.SecondaryButton {
                text: qsTr("Cancel")
                onClicked: root.close()
            }

            Controls.PrimaryButton {
                text: root.sending ? qsTr("Sending...") : qsTr("Send Feedback")
                enabled: root.canSubmit
                onClicked: {
                    root.errorMessage = ""
                    root.sending = true
                    root.feedbackService.send_feedback(
                        root.selectedTopic,
                        root.content.trim(),
                        root.email.trim(),
                        root.hasValidLicense ? root.licenseManager.getCloudToken() : ""
                    )
                }
            }
        }
        }  // ColumnLayout
    }  // Item (contentItem)
}
