import QtQuick 2.15
import QtQuick.Controls 2.15
import QtQuick.Controls.impl 2.15 as ControlsImpl
import QtQuick.Layouts 1.15
import components.media 1.0
import styles 1.0

Item {
    id: root
    property string label: "Track"
    property int trackIndex: 0
    property color trackColor: Theme.accent
    property bool muted: false
    property var waveformData: []
    property string errorMessage: ""
    property bool loading: false
    property int labelWidth: 80
    property color backgroundColor: Theme.timelineBackground
    property color labelBackground: Theme.surfaceStrong
    // Visibility of the waveform area within the track
    property bool showWaveform: true
    property int trackCount: 1

    signal trackNameChanged(int index, string name)
    signal trackMuteToggled(int index, bool muted)

    implicitHeight: 72

    readonly property color waveformColor: muted
                                      ? Qt.rgba(trackColor.r, trackColor.g, trackColor.b, 0.35)
                                      : trackColor
    readonly property color waveformPeakColor: muted
                                      ? Qt.rgba(trackColor.r, trackColor.g, trackColor.b, 0.2)
                                      : Qt.lighter(trackColor, 1.8)

    Rectangle {
        id: labelPanel
        width: root.labelWidth
        anchors.top: parent.top
        anchors.bottom: parent.bottom
        color: root.labelBackground

        ColumnLayout {
            anchors.fill: parent
            anchors.margins: Theme.spacingSm
            spacing: Theme.spacingXs

            // Editable track label
            Item {
                Layout.fillWidth: true
                Layout.preferredHeight: 20

                TextInput {
                    id: trackLabelInput
                    anchors.fill: parent
                    text: root.label
                    color: root.muted ? Theme.textSecondary : Theme.textPrimary
                    font.pixelSize: 14
                    clip: true
                    selectByMouse: true
                    selectionColor: Theme.accent
                    selectedTextColor: Theme.textOnAccent

                    property string originalText: ""

                    Component.onCompleted: cursorPosition = 0

                    onActiveFocusChanged: {
                        if (activeFocus) {
                            originalText = text
                            selectAll()
                        } else {
                            cursorPosition = 0
                        }
                    }

                    onAccepted: {
                        if (text.trim() !== "" && text !== originalText) {
                            root.trackNameChanged(root.trackIndex, text.trim())
                        } else if (text.trim() === "") {
                            text = originalText
                        }
                        focus = false
                    }

                    Keys.onEscapePressed: {
                        text = originalText
                        focus = false
                    }

                    // Show cursor on hover to indicate editable
                    MouseArea {
                        anchors.fill: parent
                        cursorShape: Qt.IBeamCursor
                        onClicked: function(mouse) {
                            trackLabelInput.forceActiveFocus()
                            mouse.accepted = true  // Prevent propagation to timeline
                        }
                    }
                }
            }

            // Mute button - only show when multiple tracks exist
            RoundButton {
                visible: root.trackCount > 1
                Layout.preferredWidth: 28
                Layout.preferredHeight: 28
                padding: 0
                flat: true
                display: AbstractButton.IconOnly
                icon.source: root.muted ? Theme.iconUrl("volume-off") : Theme.iconUrl("volume")
                icon.width: 16
                icon.height: 16
                icon.color: root.muted ? Theme.textSecondary : Theme.textPrimary
                onClicked: root.trackMuteToggled(root.trackIndex, !root.muted)
                ToolTip.visible: hovered
                ToolTip.delay: 300
                ToolTip.text: root.muted ? qsTr("Unmute track") : qsTr("Mute track")
            }

            Item {
                Layout.fillWidth: true
                Layout.fillHeight: true
            }

            Rectangle {
                id: errorChip
                Layout.alignment: Qt.AlignLeft
                Layout.preferredHeight: 24
                visible: root.errorMessage.length > 0
                radius: 12
                color: Qt.rgba(Theme.statusDanger.r, Theme.statusDanger.g, Theme.statusDanger.b, Theme.isDark ? 0.22 : 0.1)
                border.color: Theme.statusDanger
                border.width: 1
                readonly property bool hasText: root.errorMessage.length > 0

                RowLayout {
                    anchors.fill: parent
                    anchors.margins: Theme.spacingXs
                    spacing: Theme.spacingXs

                    Item {
                        Layout.preferredWidth: 16
                        Layout.preferredHeight: 16

                        ControlsImpl.IconImage {
                            anchors.fill: parent
                            source: Theme.iconUrl("alert-triangle")
                            color: Theme.statusDanger
                        }
                    }

                    Label {
                        Layout.alignment: Qt.AlignVCenter
                        Layout.fillWidth: true
                        text: root.errorMessage
                        elide: Text.ElideRight
                        font.pixelSize: 12
                        color: Theme.statusDanger
                    }
                }

                MouseArea {
                    id: errorHover
                    anchors.fill: parent
                    hoverEnabled: true
                    cursorShape: Qt.PointingHandCursor
                    acceptedButtons: Qt.NoButton
                }

                ToolTip.visible: errorHover.containsMouse
                ToolTip.text: root.errorMessage
                ToolTip.delay: 200
            }
        }
    }

    WaveformArea {
        id: waveform
        anchors.top: parent.top
        anchors.bottom: parent.bottom
        anchors.left: labelPanel.right
        anchors.right: parent.right
        anchors.margins: 0
        visible: root.showWaveform
        waveformData: root.waveformData
        fillColor: root.waveformColor
        peakColor: root.waveformPeakColor
        backgroundColor: root.backgroundColor
        borderColor: Qt.rgba(0, 0, 0, 0)
        opacity: root.errorMessage.length > 0 ? 0.25 : 1.0

        Text {
            anchors.centerIn: parent
            visible: root.loading && (!root.waveformData || root.waveformData.length === 0) && root.errorMessage.length === 0
            text: qsTr("Loading…")
            color: Theme.textSecondary
            font.pixelSize: 13
        }
    }
}
