pragma ComponentBehavior: Bound

import QtQuick 2.15
import QtQuick.Controls 2.15
import styles 1.0

Item {
    id: root

    property var analysisController: null
    property var backend: null
    property var hintManager: null
    property real laneWidth: 100
    property real timeOffset: 0.0
    property real secondsPerPixel: 0.05

    // Selection drag state (mirrors AudioTimeline's interactionArea logic)
    property bool _selectionActive: false
    property bool _selectionStarted: false
    property real _selectionAnchor: 0
    property real _pressX: 0

    signal selectionChanged(real start, real end)
    signal playheadChanged(real time)

    height: visible ? 20 : 0
    visible: analysisController && analysisController.blackVisualizationVisible

    // Hint target covering just the pills content (not the label area)
    Item {
        id: pillsContentArea
        objectName: "pillsContentArea"
        anchors.left: parent.left
        anchors.leftMargin: root.laneWidth
        anchors.right: parent.right
        anchors.top: parent.top
        anchors.bottom: parent.bottom
        // No children - just a coordinate reference for hints
    }

    // Combined model: black segments with isBlack=true, white with isBlack=false
    readonly property var combinedSegments: {
        if (!analysisController) return []
        var result = []
        var blackSegs = analysisController.blackSegments || []
        var whiteSegs = analysisController.whiteSegments || []
        for (var i = 0; i < blackSegs.length; i++) {
            result.push({ segment: blackSegs[i], isBlack: true })
        }
        for (var j = 0; j < whiteSegs.length; j++) {
            result.push({ segment: whiteSegs[j], isBlack: false })
        }
        return result
    }

    function timeToX(timeValue) {
        return (timeValue - root.timeOffset) / root.secondsPerPixel + root.laneWidth
    }

    function xToTime(position) {
        var local = position - root.laneWidth
        return root.timeOffset + local * root.secondsPerPixel
    }

    function formatDuration(seconds) {
        if (seconds < 1) {
            return (seconds * 1000).toFixed(0) + "ms"
        } else if (seconds < 60) {
            return seconds.toFixed(1) + "s"
        } else {
            var mins = Math.floor(seconds / 60)
            var secs = (seconds % 60).toFixed(0)
            return mins + ":" + (secs < 10 ? "0" : "") + secs
        }
    }

    // Find the segment that contains the given time
    function findSegmentAtTime(time) {
        for (var i = 0; i < root.combinedSegments.length; i++) {
            var seg = root.combinedSegments[i]
            if (time >= seg.segment[0] && time <= seg.segment[1]) {
                return seg
            }
        }
        return null
    }

    // Background MouseArea for drag selection (behind pills)
    MouseArea {
        id: dragArea
        anchors.fill: parent
        anchors.leftMargin: root.laneWidth
        z: -1  // Behind pills

        onPressed: function(mouse) {
            root._selectionActive = true
            root._selectionStarted = false
            root._pressX = mouse.x + root.laneWidth
            root._selectionAnchor = root.xToTime(root._pressX)
        }

        onPositionChanged: function(mouse) {
            if (!root._selectionActive) return

            var currentX = mouse.x + root.laneWidth
            var dragDistance = Math.abs(currentX - root._pressX)

            if (!root._selectionStarted && dragDistance > 5) {
                root._selectionStarted = true
            }

            if (root._selectionStarted && root.backend) {
                var clampedX = Math.max(root.laneWidth, Math.min(currentX, root.width))
                var time = root.xToTime(clampedX)
                root.backend.setSelection(root._selectionAnchor, time)
                root.selectionChanged(Math.min(root._selectionAnchor, time), Math.max(root._selectionAnchor, time))
            }
        }

        onReleased: function(mouse) {
            if (!root._selectionStarted && root.backend) {
                // Quick click - seek
                var time = root.xToTime(mouse.x + root.laneWidth)
                root.backend.setPlayheadTime(time)
                root.playheadChanged(time)
            } else if (root._selectionStarted && root.backend) {
                // Finalize selection
                var clampedX = Math.max(root.laneWidth, Math.min(mouse.x + root.laneWidth, root.width))
                var time = root.xToTime(clampedX)
                root.backend.setSelection(root._selectionAnchor, time)
            }
            root._selectionActive = false
            root._selectionStarted = false
        }

        onCanceled: {
            root._selectionActive = false
            root._selectionStarted = false
        }
    }

    Repeater {
        model: root.combinedSegments
        delegate: Rectangle {
            id: pill
            required property var modelData
            required property int index

            readonly property real startTime: modelData.segment[0]
            readonly property real endTime: modelData.segment[1]
            readonly property bool isBlack: modelData.isBlack
            readonly property real rawX: root.timeToX(startTime)
            readonly property real rawRight: root.timeToX(endTime)

            visible: rawRight > root.laneWidth && rawX < root.width
            x: Math.max(root.laneWidth, rawX)
            width: Math.max(2, rawRight - x)
            y: 2
            height: root.height - 4
            radius: 3
            color: isBlack ? Theme.blackPillColor : Theme.whitePillColor
            border.color: isBlack ? Theme.blackPillBorder : Theme.whitePillBorder
            border.width: 1

            MouseArea {
                id: pillMouseArea
                anchors.fill: parent
                cursorShape: Qt.PointingHandCursor
                hoverEnabled: true

                property real _pressX: 0
                property bool _isDragging: false
                // Store the original pill's time range for drag snapping
                property real _originStart: 0
                property real _originEnd: 0

                onPressed: function(mouse) {
                    _pressX = mouse.x
                    _isDragging = false
                    _originStart = pill.startTime
                    _originEnd = pill.endTime
                    root._selectionActive = true
                    root._selectionStarted = false
                    // Dismiss hint on press
                    if (root.hintManager) {
                        root.hintManager.dismissHint("black_white_pills")
                    }
                }

                onPositionChanged: function(mouse) {
                    if (!pressed) return
                    var dragDistance = Math.abs(mouse.x - _pressX)
                    if (!_isDragging && dragDistance > 5) {
                        _isDragging = true
                        root._selectionStarted = true
                    }
                    if (_isDragging && root.backend) {
                        // Find segment at current mouse position and snap to boundaries
                        var globalX = pill.x + mouse.x
                        var clampedX = Math.max(root.laneWidth, Math.min(globalX, root.width))
                        var time = root.xToTime(clampedX)
                        var targetSeg = root.findSegmentAtTime(time)

                        var selStart = _originStart
                        var selEnd = _originEnd
                        if (targetSeg) {
                            // Extend selection to include both original pill and target pill
                            selStart = Math.min(_originStart, targetSeg.segment[0])
                            selEnd = Math.max(_originEnd, targetSeg.segment[1])
                        } else {
                            // No segment at cursor - extend to cursor time
                            selStart = Math.min(_originStart, time)
                            selEnd = Math.max(_originEnd, time)
                        }
                        root.backend.setSelection(selStart, selEnd)
                        root.selectionChanged(selStart, selEnd)
                    }
                }

                onReleased: function(mouse) {
                    if (_isDragging) {
                        // Final snap to segment boundaries
                        if (root.backend) {
                            var globalX = pill.x + mouse.x
                            var clampedX = Math.max(root.laneWidth, Math.min(globalX, root.width))
                            var time = root.xToTime(clampedX)
                            var targetSeg = root.findSegmentAtTime(time)

                            var selStart = _originStart
                            var selEnd = _originEnd
                            if (targetSeg) {
                                selStart = Math.min(_originStart, targetSeg.segment[0])
                                selEnd = Math.max(_originEnd, targetSeg.segment[1])
                            } else {
                                selStart = Math.min(_originStart, time)
                                selEnd = Math.max(_originEnd, time)
                            }
                            root.backend.setSelection(selStart, selEnd)
                        }
                    } else if (root.backend) {
                        // Click - select this segment
                        if (mouse.modifiers & Qt.ControlModifier || mouse.modifiers & Qt.ShiftModifier) {
                            var currentSel = root.backend.selectionRange
                            if (currentSel && currentSel.length === 2) {
                                var newStart = Math.min(currentSel[0], pill.startTime)
                                var newEnd = Math.max(currentSel[1], pill.endTime)
                                root.backend.setSelection(newStart, newEnd)
                            } else {
                                root.backend.setSelection(pill.startTime, pill.endTime)
                            }
                        } else {
                            root.backend.setSelection(pill.startTime, pill.endTime)
                        }
                    }
                    _isDragging = false
                    root._selectionActive = false
                    root._selectionStarted = false
                }

                onCanceled: {
                    _isDragging = false
                    root._selectionActive = false
                    root._selectionStarted = false
                }
            }

            ToolTip {
                visible: pillMouseArea.containsMouse
                text: (pill.isBlack ? "Black: " : "Non-black: ") + root.formatDuration(pill.endTime - pill.startTime)
                delay: 500
            }
        }
    }
}
