pragma ComponentBehavior: Bound

import QtQuick 2.15
import QtQuick.Controls 2.15
import QtQuick.Layouts 1.15
import styles 1.0

Rectangle {
    id: root

    property string message: ""
    property string detail: ""  // Optional second line with more detail
    property string toastType: "info"  // "success", "error", "warning", "info"
    property int duration: 3000  // Auto-dismiss time in ms

    signal dismissed()

    implicitWidth: contentRow.implicitWidth + Theme.spacingLg * 2
    implicitHeight: Math.max(44, contentRow.implicitHeight + Theme.spacingMd * 2)
    radius: Theme.panelRadius

    color: {
        switch (toastType) {
        case "success":
            return Theme.isDark ? "#2d5a3d" : "#d4edda"
        case "error":
            return Theme.isDark ? "#5a2d2d" : "#f8d7da"
        case "warning":
            return Theme.isDark ? "#5a4a2d" : "#fff3cd"
        default:
            return Theme.surfaceRaised
        }
    }

    border.color: {
        switch (toastType) {
        case "success":
            return Theme.isDark ? "#4ade80" : "#28a745"
        case "error":
            return Theme.isDark ? "#ff8787" : "#dc3545"
        case "warning":
            return Theme.isDark ? "#fbbf24" : "#ffc107"
        default:
            return Theme.surfaceBorder
        }
    }
    border.width: 1

    RowLayout {
        id: contentRow
        anchors.centerIn: parent
        spacing: Theme.spacingSm

        // Icon
        Label {
            text: {
                switch (root.toastType) {
                case "success":
                    return "\u2713"  // Checkmark
                case "error":
                    return "\u2717"  // X mark
                case "warning":
                    return "\u26A0"  // Warning sign
                default:
                    return "\u2139"  // Info
                }
            }
            font.pixelSize: 16
            font.bold: true
            color: {
                switch (root.toastType) {
                case "success":
                    return Theme.isDark ? "#4ade80" : "#28a745"
                case "error":
                    return Theme.isDark ? "#ff8787" : "#dc3545"
                case "warning":
                    return Theme.isDark ? "#fbbf24" : "#ffc107"
                default:
                    return Theme.textPrimary
                }
            }
        }

        // Message (title + optional detail)
        ColumnLayout {
            spacing: 2
            Layout.maximumWidth: 400

            Label {
                text: root.message
                font.pixelSize: 13
                color: Theme.textPrimary
                elide: Text.ElideRight
                Layout.fillWidth: true
            }

            Label {
                visible: root.detail !== ""
                text: root.detail
                font.pixelSize: 11
                color: Theme.textSecondary
                elide: Text.ElideRight
                Layout.fillWidth: true
            }
        }

        // Close button
        Rectangle {
            Layout.preferredWidth: 20
            Layout.preferredHeight: 20
            radius: 10
            color: closeArea.containsMouse ? Theme._withAlpha(Theme.textPrimary, 0.1) : "transparent"

            Label {
                anchors.centerIn: parent
                text: "\u00d7"  // Multiplication sign (X)
                font.pixelSize: 14
                color: Theme.textSecondary
            }

            MouseArea {
                id: closeArea
                anchors.fill: parent
                hoverEnabled: true
                cursorShape: Qt.PointingHandCursor
                onClicked: root.dismissed()
            }
        }
    }

    // Auto-dismiss timer
    Timer {
        id: dismissTimer
        interval: root.duration
        running: root.visible && root.duration > 0
        onTriggered: root.dismissed()
    }

    // Slide in from right animation
    x: 300  // Start off-screen to the right
    Component.onCompleted: x = 0

    Behavior on x {
        NumberAnimation { duration: 250; easing.type: Easing.OutCubic }
    }
}
