pragma ComponentBehavior: Bound
import QtQuick 2.15
import QtQuick.Controls 2.15
import features.cut 1.0
import styles 1.0

Item {
    id: timeline
    required property var backend
    property var analysisController: null
    property var hintManager: null
    property bool enabled: true
    property string emptyStateText: ""
    property real laneWidth: 100
    property real trackHeight: 72
    property real trackSpacing: 8
    property real rulerHeight: 28
    property real secondsPerPixel: 0.05
    property real timeOffset: 0.0
    // Toggle visibility of per-track waveforms
    property bool showWaveforms: true
    property color rulerTextColor: Theme.textSecondary
    property color tickColor: Theme.timelineTick
    property color playheadColor: Theme.timelinePlayhead
    property color selectionFill: Theme.selectionHighlight
    property color selectionBorder: Theme.timelineSelectionBorder
    onTickColorChanged: {
        rulerCanvas.requestPaint()
        tickOverlayCanvas.requestPaint()
    }
    onRulerTextColorChanged: rulerCanvas.requestPaint()

    readonly property real waveformWidth: Math.max(1, timeline.width - timeline.laneWidth)
    readonly property real visibleDuration: timeline.secondsPerPixel * timeline.waveformWidth
    readonly property real totalDuration: backend ? backend.duration : 0
    // Height of the black/white pills row (20 when visible, 0 when hidden)
    readonly property real pillsRowHeight: blackWhitePills.visible ? blackWhitePills.height : 0
    readonly property var trackData: backend ? backend.trackData : []
    readonly property var selectionRange: backend ? backend.selectionRange : []
    readonly property var loopRange: backend ? backend.loopRange : []
    readonly property var cutRanges: backend ? backend.cutRanges : []
    readonly property int trackCount: timeline.trackData ? timeline.trackData.length : 0
    readonly property real contentHeight: {
        // Add spacing between tracks and include the ruler region
        var count = Math.max(1, trackCount)
        var tracks = count * timeline.trackHeight
        var spacing = Math.max(0, (count - 1) * timeline.trackSpacing)
        var bottomPadding = 6
        var pillsHeight = timeline.pillsRowHeight
        return timeline.rulerHeight + pillsHeight + tracks + spacing + bottomPadding
    }

    implicitHeight: contentHeight

    signal selectionChanged(real start, real end)
    signal playheadChanged(real time)

    property bool _suppressUpdate: false
    property real _selectionAnchor: 0
    property bool _selectionActive: false
    property bool _selectionMoved: false
    property bool _selectionStarted: false  // True once drag threshold crossed
    property real _pressX: 0  // Initial click X position for drag threshold
    property bool _panActive: false
    property real _lastPanX: 0

    // Throttling for seek events during ruler scrubbing
    property real _pendingSeekTime: -1

    // Throttle seek emissions to ~60fps to prevent overwhelming audio engine
    Timer {
        id: seekThrottleTimer
        interval: 16  // ~60fps
        repeat: false
        onTriggered: {
            if (timeline._pendingSeekTime >= 0 && timeline.backend) {
                timeline.backend.setPlayheadTime(timeline._pendingSeekTime)
                timeline.playheadChanged(timeline._pendingSeekTime)
                timeline._pendingSeekTime = -1
            }
        }
    }

    onVisibleChanged: {
        if (visible && backend) {
            // Sync zoom/pan state FROM backend when becoming visible
            timeline._suppressUpdate = true
            timeline.timeOffset = backend.timeOffset
            timeline.secondsPerPixel = backend.secondsPerPixel
            timeline._suppressUpdate = false
            // Then update view with correct width
            requestViewUpdate()
        }
    }

    function timeToX(timeValue) {
        return (timeValue - timeline.timeOffset) / timeline.secondsPerPixel + timeline.laneWidth
    }

    function xToTime(position) {
        var local = position - timeline.laneWidth
        return timeline.timeOffset + local * timeline.secondsPerPixel
    }

    function clampTimeOffset(offset) {
        if (timeline.totalDuration <= 0) {
            return 0
        }
        var visible = timeline.visibleDuration
        if (visible >= timeline.totalDuration) {
            return 0
        }
        var minOffset = 0
        var maxOffset = Math.max(0, timeline.totalDuration - visible)
        if (offset < minOffset) {
            return minOffset
        }
        if (offset > maxOffset) {
            return maxOffset
        }
        return offset
    }

    function fitToDuration() {
        if (!backend || timeline.waveformWidth <= 0) {
            return
        }
        if (timeline.totalDuration <= 0) {
            timeline.secondsPerPixel = 0.05
            timeline.timeOffset = 0
            return
        }
        timeline._suppressUpdate = true
        timeline.secondsPerPixel = timeline.totalDuration / timeline.waveformWidth
        timeline.timeOffset = 0
        timeline._suppressUpdate = false
        requestViewUpdate()
    }

    function centerOnRange(startTime, endTime) {
        if (!backend || timeline.waveformWidth <= 0 || timeline.totalDuration <= 0) {
            return
        }

        // Normalize to valid bounds
        var rangeStart = Math.max(0, Math.min(startTime, endTime))
        var rangeEnd = Math.min(timeline.totalDuration, Math.max(startTime, endTime))
        var rangeDuration = rangeEnd - rangeStart

        if (rangeDuration <= 0) {
            return
        }

        // Add padding (20% on each side, minimum 0.5s each side)
        var padding = Math.max(0.5, rangeDuration * 0.2)
        var paddedStart = Math.max(0, rangeStart - padding)
        var paddedEnd = Math.min(timeline.totalDuration, rangeEnd + padding)
        var paddedDuration = paddedEnd - paddedStart

        // Calculate zoom level to fit the padded range
        var targetSecondsPerPixel = paddedDuration / timeline.waveformWidth

        // Clamp zoom: don't zoom in more than 0.0001 s/px, don't zoom out beyond full duration
        var minSeconds = 0.0001
        var maxSeconds = timeline.totalDuration / timeline.waveformWidth
        targetSecondsPerPixel = Math.max(minSeconds, Math.min(targetSecondsPerPixel, maxSeconds))

        // Calculate visible duration with NEW zoom level
        var newVisibleDuration = targetSecondsPerPixel * timeline.waveformWidth

        // Calculate offset to center the range
        var rangeCenter = (rangeStart + rangeEnd) / 2
        var targetOffset = rangeCenter - newVisibleDuration / 2

        // Clamp offset using NEW visible duration (can't use clampTimeOffset - it uses old zoom)
        if (newVisibleDuration >= timeline.totalDuration) {
            targetOffset = 0
        } else {
            var maxOffset = Math.max(0, timeline.totalDuration - newVisibleDuration)
            targetOffset = Math.max(0, Math.min(targetOffset, maxOffset))
        }

        // Apply changes
        timeline._suppressUpdate = true
        timeline.secondsPerPixel = targetSecondsPerPixel
        timeline.timeOffset = targetOffset
        timeline._suppressUpdate = false
        requestViewUpdate()
    }

    function requestViewUpdate() {
        if (!backend || timeline._suppressUpdate || timeline.waveformWidth <= 0) {
            return
        }
        if (!timeline.visible) {
            return  // Don't update when not visible to avoid conflicts with other workspaces
        }
        var changed = backend.setViewParameters(
                    Math.max(1, Math.round(timeline.waveformWidth)),
                    Math.max(1, Math.round(timeline.trackHeight)),
                    timeline.timeOffset,
                    timeline.secondsPerPixel)
        if (changed) {
            var backendOffset = backend.timeOffset
            var backendSeconds = backend.secondsPerPixel
            var needsSync = Math.abs(backendOffset - timeline.timeOffset) > 1e-6 ||
                            Math.abs(backendSeconds - timeline.secondsPerPixel) > 1e-9
            if (needsSync) {
                timeline._suppressUpdate = true
                timeline.timeOffset = backendOffset
                timeline.secondsPerPixel = backendSeconds
                timeline._suppressUpdate = false
            }
        }
    }

    function tickStep(rangeDuration) {
        var nice = [0.1, 0.2, 0.5, 1, 2, 5, 10, 15, 30, 60, 120, 300, 600, 900, 1800, 3600]
        var target = rangeDuration / 8
        for (var i = 0; i < nice.length; ++i) {
            if (nice[i] >= target) {
                return nice[i]
            }
        }
        return nice[nice.length - 1]
    }

    function formatTime(seconds, step) {
        var negative = seconds < 0
        seconds = Math.abs(seconds)
        var hrs = Math.floor(seconds / 3600)
        var mins = Math.floor((seconds % 3600) / 60)
        var secs = seconds % 60
        var text = ""
        if (hrs > 0) {
            text += String(hrs) + ":"
            text += (mins < 10 ? "0" : "") + String(mins) + ":"
        } else {
            text += String(mins) + ":"
        }
        var secsInt = Math.floor(secs)
        text += (secsInt < 10 ? "0" : "") + String(secsInt)
        // Only show milliseconds if tick step is sub-second
        if (step < 1) {
            var remainder = Math.abs(secs - secsInt)
            var ms = Math.floor(remainder * 1000)
            text += "." + ms.toString().padStart(3, "0")
        }
        return negative ? "-" + text : text
    }

    onBackendChanged: {
        if (timeline.backend) {
            timeline.requestViewUpdate()
        }
    }

    onWidthChanged: timeline.requestViewUpdate()
    onSecondsPerPixelChanged: {
        if (!timeline._suppressUpdate) {
            timeline.requestViewUpdate()
        }
        rulerCanvas.requestPaint()
        tickOverlayCanvas.requestPaint()
    }

    onTimeOffsetChanged: {
        if (!timeline._suppressUpdate) {
            timeline.requestViewUpdate()
        }
        rulerCanvas.requestPaint()
        tickOverlayCanvas.requestPaint()
    }


    Item {
        id: ruler
        anchors.left: parent.left
        anchors.right: parent.right
        anchors.top: parent.top
        height: timeline.rulerHeight

        Canvas {
            id: rulerCanvas
            anchors.fill: parent
            renderTarget: Canvas.FramebufferObject

            onPaint: {
                var ctx = getContext("2d")
                ctx.reset()
                ctx.clearRect(0, 0, width, height)

                ctx.strokeStyle = timeline.tickColor
                ctx.fillStyle = timeline.rulerTextColor
                ctx.font = "14px sans-serif"
                ctx.lineWidth = 1

                var start = timeline.timeOffset
                var end = start + timeline.visibleDuration
                var step = timeline.tickStep(timeline.visibleDuration)
                if (step <= 0) {
                    return
                }
                var firstTick = Math.floor(start / step) * step

                for (var t = firstTick; t <= end + step; t += step) {
                    var x = timeline.timeToX(t)
                    if (x < timeline.laneWidth) {
                        continue
                    }
                    if (x > width) {
                        break
                    }
                    if (timeline.waveformWidth > 120) {
                        var text = timeline.formatTime(t, step)
                        var textWidth = ctx.measureText(text).width
                        ctx.fillText(text, x - textWidth / 2, 18)
                    }
                }
            }

            onVisibleChanged: if (visible) requestPaint()
        }

        // Scrubbing mouse area for ruler
        MouseArea {
            id: rulerMouseArea
            anchors.fill: parent
            anchors.leftMargin: timeline.laneWidth  // Don't cover track label area
            hoverEnabled: true

            property real hoverTime: -1

            onPressed: function(ev) {
                var time = timeline.xToTime(ev.x + timeline.laneWidth)
                time = Math.max(0, Math.min(time, timeline.totalDuration))
                timeline._pendingSeekTime = time
                seekThrottleTimer.start()
            }

            onPositionChanged: function(ev) {
                // Update hover time for tooltip
                var time = timeline.xToTime(ev.x + timeline.laneWidth)
                hoverTime = Math.max(0, Math.min(time, timeline.totalDuration))

                if (!pressed) return
                timeline._pendingSeekTime = hoverTime
                seekThrottleTimer.start()
            }

            onExited: {
                hoverTime = -1
            }

            ToolTip {
                visible: rulerMouseArea.containsMouse && timeline.totalDuration > 0 && rulerMouseArea.hoverTime >= 0
                text: timeline.formatTime(rulerMouseArea.hoverTime, timeline.tickStep(timeline.visibleDuration))
                delay: 100
                x: Math.max(0, Math.min(rulerMouseArea.mouseX - width / 2, rulerMouseArea.width - width))
                y: -height - 2
            }
        }
    }

    // Tick lines spanning from ruler into track area
    Canvas {
        id: tickOverlayCanvas
        anchors.left: parent.left
        anchors.right: parent.right
        anchors.top: ruler.bottom
        anchors.topMargin: -6
        anchors.bottom: parent.bottom
        z: 8
        renderTarget: Canvas.FramebufferObject
        antialiasing: false
        smooth: false

        onPaint: {
            var ctx = getContext("2d")
            ctx.reset()
            ctx.clearRect(0, 0, width, height)

            var start = timeline.timeOffset
            var end = start + timeline.visibleDuration
            var step = timeline.tickStep(timeline.visibleDuration)
            if (step <= 0) {
                return
            }

            ctx.lineWidth = 1
            ctx.lineCap = "butt"
            var tickColor = Theme.timelineTick
            ctx.strokeStyle = Qt.rgba(tickColor.r, tickColor.g, tickColor.b, 0.3)

            var firstTick = Math.floor(start / step) * step
            for (var t = firstTick; t <= end + step; t += step) {
                var x = timeline.timeToX(t)
                if (x < timeline.laneWidth) {
                    continue
                }
                if (x > width) {
                    break
                }
                ctx.beginPath()
                ctx.moveTo(x + 0.5, 0.5)
                ctx.lineTo(x + 0.5, height)
                ctx.stroke()
            }
        }

        onVisibleChanged: if (visible) requestPaint()
        onWidthChanged: requestPaint()
        onHeightChanged: requestPaint()
    }

    // Single playhead component with triangle and line - merged for pixel-perfect alignment
    Item {
        id: playheadContainer
        anchors.left: parent.left
        anchors.leftMargin: timeline.laneWidth
        anchors.right: parent.right
        anchors.top: parent.top
        anchors.topMargin: timeline.rulerHeight - 8  // Position so triangle sits at bottom of ruler
        anchors.bottom: parent.bottom
        clip: true
        z: 20

        Canvas {
            id: playheadCanvas
            readonly property real rawX: timeline.timeToX(timeline.backend ? timeline.backend.playheadTime : 0) - timeline.laneWidth
            visible: timeline.backend && timeline.backend.playheadTime >= -timeline.totalDuration && rawX >= -10
            x: rawX - 6  // Center the canvas on playhead (triangle tip and line center both at position 6)
            y: 0
            width: 12
            height: parent.height

            property color playheadColor: timeline.playheadColor
            onPlayheadColorChanged: requestPaint()
            onHeightChanged: requestPaint()

            onPaint: {
                var ctx = getContext('2d')
                ctx.clearRect(0, 0, width, height)
                ctx.fillStyle = playheadColor

                // Triangle at top (tip at center, pointing down)
                var triangleHeight = 8
                ctx.beginPath()
                ctx.moveTo(0, 0)
                ctx.lineTo(width, 0)
                ctx.lineTo(width / 2, triangleHeight)
                ctx.closePath()
                ctx.fill()

                // Line from triangle tip to bottom (2px wide, centered at width/2)
                var lineX = (width - 2) / 2
                ctx.fillRect(lineX, triangleHeight - 1, 2, height - triangleHeight + 1)
            }
        }
    }

    // Black/white segment pills row (visible when black detection visualization is enabled)
    BlackWhitePillsRow {
        id: blackWhitePills
        objectName: "blackWhitePillsRow"
        anchors.left: parent.left
        anchors.right: parent.right
        anchors.top: ruler.bottom
        laneWidth: timeline.laneWidth
        timeOffset: timeline.timeOffset
        secondsPerPixel: timeline.secondsPerPixel
        analysisController: timeline.analysisController
        backend: timeline.backend
        hintManager: timeline.hintManager
    }

    Item {
        id: trackArea
        anchors.left: parent.left
        anchors.right: parent.right
        anchors.leftMargin: 1
        anchors.rightMargin: 1
        anchors.top: blackWhitePills.visible ? blackWhitePills.bottom : ruler.bottom
        anchors.bottom: parent.bottom
        clip: true

        // Hint target covering just the waveform content (not labels)
        Item {
            id: waveformContentArea
            objectName: "waveformContentArea"
            anchors.left: parent.left
            anchors.leftMargin: timeline.laneWidth - 1
            anchors.right: parent.right
            anchors.top: parent.top
            anchors.bottom: parent.bottom
            // No children - just a coordinate reference for hints
        }

        Column {
            id: trackColumn
            anchors.left: parent.left
            anchors.right: parent.right
            anchors.top: parent.top
            spacing: timeline.trackSpacing
            Repeater {
                model: timeline.trackData
                delegate: TimelineTrack {
                    required property var modelData
                    required property int index
                    width: timeline.width
                    height: timeline.trackHeight
                    labelWidth: timeline.laneWidth
                    label: modelData.label
                    trackIndex: modelData.index
                    trackColor: modelData.color
                    muted: modelData.muted
                    showWaveform: timeline.showWaveforms
                    waveformData: modelData.waveform
                    errorMessage: modelData.error
                    loading: modelData.loading
                    trackCount: timeline.trackData.length

                    onTrackNameChanged: function(idx, name) {
                        if (timeline.backend) {
                            timeline.backend.setTrackName(idx, name)
                        }
                    }

                    onTrackMuteToggled: function(idx, muted) {
                        if (timeline.backend) {
                            timeline.backend.setTrackMuted(idx, muted)
                        }
                    }
                }
            }
        }

        Repeater {
            model: timeline.cutRanges
            delegate: Rectangle {
                id: cutDelegate
                required property var modelData
                z: 6
                visible: rawRight > timeline.laneWidth  // Hide when entirely off-screen left
                color: Theme.timelineCutOverlay
                radius: 0
                anchors.top: parent.top
                anchors.bottom: parent.bottom
                readonly property real rawX: Math.min(timeline.timeToX(modelData[0]), timeline.timeToX(modelData[1]))
                readonly property real rawRight: Math.max(timeline.timeToX(modelData[0]), timeline.timeToX(modelData[1]))
                x: Math.max(timeline.laneWidth, rawX)
                width: Math.max(0, rawRight - x)

                // Top border
                Rectangle {
                    anchors.left: parent.left
                    anchors.right: parent.right
                    anchors.top: parent.top
                    height: 1
                    color: Theme.timelineCutOverlayBorder
                }
                // Bottom border
                Rectangle {
                    anchors.left: parent.left
                    anchors.right: parent.right
                    anchors.bottom: parent.bottom
                    height: 1
                    color: Theme.timelineCutOverlayBorder
                }
                // Left border (only if not clipped)
                Rectangle {
                    visible: cutDelegate.rawX >= timeline.laneWidth
                    anchors.left: parent.left
                    anchors.top: parent.top
                    anchors.bottom: parent.bottom
                    width: 1
                    color: Theme.timelineCutOverlayBorder
                }
                // Right border (only if not clipped on right)
                Rectangle {
                    visible: cutDelegate.rawRight <= trackArea.width
                    anchors.right: parent.right
                    anchors.top: parent.top
                    anchors.bottom: parent.bottom
                    width: 1
                    color: Theme.timelineCutOverlayBorder
                }
            }
        }

        // A/B loop range overlay (independent of selection)
        Rectangle {
            id: loopRect
            visible: timeline.loopRange.length === 2 && rawRight > timeline.laneWidth
            color: Theme.timelineLoopFill
            z: 9
            anchors.top: parent.top
            anchors.bottom: parent.bottom
            readonly property real rawX: Math.min(timeline.timeToX(timeline.loopRange[0]), timeline.timeToX(timeline.loopRange[1]))
            readonly property real rawRight: Math.max(timeline.timeToX(timeline.loopRange[0]), timeline.timeToX(timeline.loopRange[1]))
            x: Math.max(timeline.laneWidth, rawX)
            width: Math.max(0, rawRight - x)

            // Top border
            Rectangle {
                anchors.left: parent.left
                anchors.right: parent.right
                anchors.top: parent.top
                height: 1
                color: Theme.timelineLoopBorder
            }
            // Bottom border
            Rectangle {
                anchors.left: parent.left
                anchors.right: parent.right
                anchors.bottom: parent.bottom
                height: 1
                color: Theme.timelineLoopBorder
            }
            // Left border (only if not clipped)
            Rectangle {
                visible: loopRect.rawX >= timeline.laneWidth
                anchors.left: parent.left
                anchors.top: parent.top
                anchors.bottom: parent.bottom
                width: 1
                color: Theme.timelineLoopBorder
            }
            // Right border (only if not clipped on right)
            Rectangle {
                visible: loopRect.rawRight <= trackArea.width
                anchors.right: parent.right
                anchors.top: parent.top
                anchors.bottom: parent.bottom
                width: 1
                color: Theme.timelineLoopBorder
            }
        }

        Rectangle {
            id: selectionRect
            visible: timeline.selectionRange.length === 2 && rawRight > timeline.laneWidth
            color: timeline.selectionFill
            z: 10
            anchors.top: parent.top
            anchors.bottom: parent.bottom
            readonly property real rawX: Math.min(timeline.timeToX(timeline.selectionRange[0]), timeline.timeToX(timeline.selectionRange[1]))
            readonly property real rawRight: Math.max(timeline.timeToX(timeline.selectionRange[0]), timeline.timeToX(timeline.selectionRange[1]))
            x: Math.max(timeline.laneWidth, rawX)
            width: Math.max(0, rawRight - x)

            // Top border
            Rectangle {
                anchors.left: parent.left
                anchors.right: parent.right
                anchors.top: parent.top
                height: 1
                color: timeline.selectionBorder
            }
            // Bottom border
            Rectangle {
                anchors.left: parent.left
                anchors.right: parent.right
                anchors.bottom: parent.bottom
                height: 1
                color: timeline.selectionBorder
            }
            // Left border (only if not clipped)
            Rectangle {
                visible: selectionRect.rawX >= timeline.laneWidth
                anchors.left: parent.left
                anchors.top: parent.top
                anchors.bottom: parent.bottom
                width: 1
                color: timeline.selectionBorder
            }
            // Right border (only if not clipped on right)
            Rectangle {
                visible: selectionRect.rawRight <= trackArea.width
                anchors.right: parent.right
                anchors.top: parent.top
                anchors.bottom: parent.bottom
                width: 1
                color: timeline.selectionBorder
            }
        }

        // A/B loop markers (A and B) drawn as vertical lines with a small triangle at the top
        Item {
            id: loopMarkers
            visible: timeline.loopRange.length === 2
            anchors.fill: parent
            z: 13

            readonly property real aTime: timeline.loopRange.length === 2 ? Math.min(timeline.loopRange[0], timeline.loopRange[1]) : 0
            readonly property real bTime: timeline.loopRange.length === 2 ? Math.max(timeline.loopRange[0], timeline.loopRange[1]) : 0
            readonly property real aX: timeline.timeToX(aTime)
            readonly property real bX: timeline.timeToX(bTime)

            // A marker line
            Rectangle {
                x: loopMarkers.aX - 1
                width: 2
                color: Theme.timelineLoopBorder
                anchors.top: parent.top
                anchors.bottom: parent.bottom
            }
            // B marker line
            Rectangle {
                x: loopMarkers.bX - 1
                width: 2
                color: Theme.timelineLoopBorder
                anchors.top: parent.top
                anchors.bottom: parent.bottom
            }
        }

        MouseArea {
            id: interactionArea
            anchors.fill: parent
            acceptedButtons: Qt.LeftButton | Qt.RightButton | Qt.MiddleButton
            hoverEnabled: true

            onPressed: function(mouse) {
                // Ignore clicks in the track label panel area
                if (mouse.x < timeline.laneWidth) {
                    mouse.accepted = false
                    return
                }
                // Ignore clicks that are actually in the pills row above us
                if (blackWhitePills.visible) {
                    var posInTimeline = mapToItem(timeline, mouse.x, mouse.y)
                    var pillsBottom = timeline.rulerHeight + blackWhitePills.height
                    if (posInTimeline.y < pillsBottom) {
                        mouse.accepted = false
                        return
                    }
                }
                if (mouse.button === Qt.RightButton || mouse.button === Qt.MiddleButton) {
                    timeline._panActive = true
                    timeline._lastPanX = mouse.x
                    mouse.accepted = true
                } else if (mouse.button === Qt.LeftButton) {
                    timeline._selectionActive = true
                    timeline._selectionMoved = false
                    timeline._selectionStarted = false  // Don't start selection until drag threshold
                    timeline._pressX = mouse.x
                    var clampedX = Math.max(timeline.laneWidth, Math.min(mouse.x, width))
                    timeline._selectionAnchor = timeline.xToTime(clampedX)
                    // DON'T call setSelection here - wait for drag threshold to preserve existing selection
                    mouse.accepted = true
                }
            }

            onPositionChanged: function(mouse) {
                if (timeline._panActive) {
                    var deltaX = mouse.x - timeline._lastPanX
                    timeline._lastPanX = mouse.x
                    timeline._suppressUpdate = true
                    timeline.timeOffset = timeline.clampTimeOffset(timeline.timeOffset - deltaX * timeline.secondsPerPixel)
                    timeline._suppressUpdate = false
                    timeline.requestViewUpdate()
                    mouse.accepted = true
                } else if (timeline._selectionActive) {
                    var dragDistance = Math.abs(mouse.x - timeline._pressX)

                    // Only start new selection after crossing 5px drag threshold
                    // This allows click-to-seek without clearing existing selection
                    if (!timeline._selectionStarted && dragDistance > 5) {
                        timeline._selectionStarted = true
                    }

                    if (timeline._selectionStarted) {
                        var clampedX = Math.max(timeline.laneWidth, Math.min(mouse.x, width))
                        var time = timeline.xToTime(clampedX)
                        if (Math.abs(time - timeline._selectionAnchor) > timeline.secondsPerPixel * 2) {
                            timeline._selectionMoved = true
                        }
                        if (timeline.backend) {
                            timeline.backend.setSelection(timeline._selectionAnchor, time)
                            timeline.selectionChanged(Math.min(timeline._selectionAnchor, time), Math.max(timeline._selectionAnchor, time))
                        }
                    }
                    mouse.accepted = true
                }
            }

            onReleased: function(mouse) {
                if ((mouse.button === Qt.RightButton || mouse.button === Qt.MiddleButton) && timeline._panActive) {
                    timeline._panActive = false
                    mouse.accepted = true
                } else if (mouse.button === Qt.LeftButton && timeline._selectionActive) {
                    var clampedX = Math.max(timeline.laneWidth, Math.min(mouse.x, width))
                    var time = timeline.xToTime(clampedX)
                    if (!timeline._selectionStarted && timeline.backend) {
                        // Quick click without drag - just seek, preserve existing selection
                        timeline.backend.setPlayheadTime(time)
                        timeline.playheadChanged(time)
                    } else if (timeline.backend) {
                        // Was dragging - finalize the new selection
                        timeline.backend.setSelection(timeline._selectionAnchor, time)
                    }
                    timeline._selectionActive = false
                    timeline._selectionMoved = false
                    timeline._selectionStarted = false
                    mouse.accepted = true
                }
            }

            onCanceled: {
                timeline._panActive = false
                timeline._selectionActive = false
                timeline._selectionMoved = false
                timeline._selectionStarted = false
            }

            onWheel: function(wheel) {
                if (!timeline.backend) {
                    return
                }

                // Horizontal scroll → pan (33% of visible width per notch)
                if (Math.abs(wheel.angleDelta.x) > 0) {
                    var panAmount = -wheel.angleDelta.x / 120 * timeline.visibleDuration * 0.33
                    timeline.timeOffset = timeline.clampTimeOffset(timeline.timeOffset + panAmount)
                    timeline.requestViewUpdate()
                }

                // Vertical scroll → zoom (or pan with Shift)
                if (wheel.angleDelta.y !== 0) {
                    if (wheel.modifiers & Qt.ShiftModifier) {
                        // Shift+wheel → pan
                        var panAmount = -wheel.angleDelta.y / 120 * timeline.visibleDuration * 0.33
                        timeline.timeOffset = timeline.clampTimeOffset(timeline.timeOffset + panAmount)
                        timeline.requestViewUpdate()
                    } else {
                        // Normal wheel → zoom
                        var factor = wheel.angleDelta.y > 0 ? 1 / 1.23 : 1.23
                        var cursorX = Math.max(timeline.laneWidth, Math.min(wheel.x, width))
                        var cursorTime = timeline.xToTime(cursorX)
                        var minSeconds = 0.0001
                        var desiredSeconds = timeline.secondsPerPixel * factor
                        var maxSeconds = timeline.totalDuration > 0 ? timeline.totalDuration / Math.max(1, timeline.waveformWidth) : desiredSeconds
                        if (timeline.totalDuration > 0) {
                            maxSeconds = Math.max(minSeconds, maxSeconds)
                        }
                        var newSeconds = Math.max(minSeconds, Math.min(desiredSeconds, maxSeconds))
                        timeline._suppressUpdate = true
                        timeline.secondsPerPixel = newSeconds
                        timeline.timeOffset = cursorTime - (cursorX - timeline.laneWidth) * timeline.secondsPerPixel
                        timeline.timeOffset = timeline.clampTimeOffset(timeline.timeOffset)
                        timeline._suppressUpdate = false
                        timeline.requestViewUpdate()
                    }
                }

                wheel.accepted = true
            }
        }

    }

    Connections {
        target: timeline.backend
        enabled: timeline.backend !== null
        function onMediaReady() {
            timeline.requestViewUpdate()
        }
        function onCenterOnRangeRequested(start, end) {
            timeline.centerOnRange(start, end)
        }
        function onViewChanged() {
            // Sync local properties from backend when it changes (e.g., via zoomIn/zoomOut slots)
            if (timeline._suppressUpdate) return
            var backendOffset = timeline.backend.timeOffset
            var backendSeconds = timeline.backend.secondsPerPixel
            var needsSync = Math.abs(backendOffset - timeline.timeOffset) > 1e-6 ||
                            Math.abs(backendSeconds - timeline.secondsPerPixel) > 1e-9
            if (needsSync) {
                timeline._suppressUpdate = true
                timeline.timeOffset = backendOffset
                timeline.secondsPerPixel = backendSeconds
                timeline._suppressUpdate = false
            }
        }
    }

    Connections {
        target: timeline.backend
        enabled: timeline.backend !== null
        function onTrackDataChanged() {
            // Trigger ruler redraw when waveform data updates
            rulerCanvas.requestPaint()
        }
    }

    // Empty state label
    Label {
        anchors.centerIn: parent
        text: timeline.emptyStateText
        color: Theme.textSecondary
        visible: !timeline.enabled && timeline.emptyStateText.length > 0
    }
}
