pragma ComponentBehavior: Bound

import QtQuick 2.15
import QtQuick.Layouts 1.15
import Qt5Compat.GraphicalEffects
import styles 1.0

Item {
    id: root

    required property var hint
    required property Item searchRoot
    required property var hintManager
    property Item headerRoot: null  // Optional: for searching ApplicationWindow header
    property Item footerRoot: null  // Optional: for searching bottom bar

    visible: root.hint !== null && targetItem !== null

    // Debug mode
    readonly property bool debug: false

    // Track which hint we found the target for (by target name, not object identity)
    property string _lastTargetName: ""
    property Item _cachedTarget: null
    property int _searchGeneration: 0  // Incremented to force re-search

    // Update cache when hint target changes
    onHintChanged: {
        var newTargetName = root.hint ? root.hint.targetObjectName : ""
        if (newTargetName !== root._lastTargetName) {
            root._lastTargetName = newTargetName
            root._cachedTarget = null  // Clear immediately
            retrySearchTimer.stop()  // Stop any pending retry from previous hint
            retrySearchTimer.attempts = 0
            if (newTargetName) {
                initialSearchTimer.restart()  // Delay search for layout to settle
            }
        }
    }

    Connections {
        target: root.hintManager || null
        function onLayoutChanged() {
            if (!root._cachedTarget) {
                return
            }
            // Don't clear cache - target is still valid, just repositioning
            // Delay the position update to let layout settle
            layoutSettleTimer.restart()
        }
    }

    // Delay position recalc after layout changes
    Timer {
        id: layoutSettleTimer
        interval: 16  // One frame
        onTriggered: root._searchGeneration++
    }

    // Small delay before initial search to let QML layout settle
    Timer {
        id: initialSearchTimer
        interval: 10
        repeat: false
        onTriggered: {
            var name = root._lastTargetName
            if (name) {
                root._cachedTarget = root.findTarget(name)
                if (!root._cachedTarget || root._cachedTarget.width === 0) {
                    retrySearchTimer.restart()
                }
            }
        }
    }

    // Retry timer for when target isn't found or has zero size (e.g., Loader not loaded yet)
    Timer {
        id: retrySearchTimer
        interval: 50
        repeat: true
        property int attempts: 0
        onTriggered: {
            attempts++
            if (attempts > 20) {  // Give up after ~1 second
                stop()
                attempts = 0
                return
            }
            var name = root.hint ? root.hint.targetObjectName : ""
            if (!name) {
                stop()
                attempts = 0
                return
            }
            var found = root.findTarget(name)
            if (found && found.width > 0 && root.parent) {
                // Also verify mapToItem returns valid coordinates
                var pos = found.mapToItem(root.parent, 0, 0)
                if (pos.x !== 0 || pos.y !== 0 || found.x !== 0 || found.y !== 0) {
                    root._cachedTarget = found
                    root._searchGeneration++  // Force binding update
                    stop()
                    attempts = 0
                }
            }
        }
    }

    // Use cached target - only updated when hint changes or retry succeeds
    readonly property Item targetItem: {
        // Depend on _searchGeneration to force re-evaluation after retry
        var gen = root._searchGeneration
        return root._cachedTarget
    }

    function findTarget(name) {
        if (!name) return null
        // Search main content first
        var result = findItemByObjectName(root.searchRoot, name)
        if (result) return result
        // Then search header if provided
        if (root.headerRoot) {
            result = findItemByObjectName(root.headerRoot, name)
            if (result) return result
        }
        // Then search footer if provided
        if (root.footerRoot) {
            result = findItemByObjectName(root.footerRoot, name)
        }
        return result
    }

    function findItemByObjectName(searchParent, name) {
        if (!searchParent || !name) return null
        if (searchParent.objectName === name) return searchParent
        for (var i = 0; i < searchParent.children.length; i++) {
            var child = searchParent.children[i]
            if (child.objectName === name) return child
        }
        for (var j = 0; j < searchParent.children.length; j++) {
            var child = searchParent.children[j]
            var result = findItemByObjectName(child, name)
            if (result) return result
            // Also search inside Loader items
            if (child.item !== undefined && child.item !== null) {
                result = findItemByObjectName(child.item, name)
                if (result) return result
            }
        }
        return null
    }

    // Positioning
    readonly property real arrowSize: 8
    readonly property real gap: 4
    readonly property real highlightPadding: 6
    readonly property real highlightRadius: 8
    readonly property real highlightBorderWidth: 2
    readonly property real highlightGlowRadius: 14
    readonly property bool showBelow: root.hint ? root.hint.position === "below" : true
    readonly property string targetAnchor: root.hint && root.hint.targetAnchor ? root.hint.targetAnchor : "center"
    readonly property bool showHighlight: root.visible
        && (!root.hint || root.hint.highlightEnabled !== false)
        && root.targetBounds.width > 0
        && root.targetBounds.height > 0

    // Computed target bounds in parent coordinates
    // Note: binds to targetItem geometry to re-evaluate when it changes
    readonly property rect targetBounds: {
        if (!root.targetItem || !root.parent) return Qt.rect(0, 0, 0, 0)
        // Force re-evaluation when target geometry changes by reading these properties
        var w = root.targetItem.width
        var h = root.targetItem.height
        var tx = root.targetItem.x
        var ty = root.targetItem.y
        // Also depend on searchGeneration to force update after retry finds target
        var gen = root._searchGeneration
        var topLeft = root.targetItem.mapToItem(root.parent, 0, 0)
        return Qt.rect(topLeft.x, topLeft.y, w, h)
    }

    // Arrow tip should point at center-top or center-bottom of target
    readonly property real arrowTipX: {
        if (root.targetAnchor === "left") {
            return root.targetBounds.x + Math.min(root.targetBounds.width * 0.15, 120)
        }
        if (root.targetAnchor === "right") {
            return root.targetBounds.x + root.targetBounds.width - Math.min(root.targetBounds.width * 0.15, 120)
        }
        return root.targetBounds.x + root.targetBounds.width / 2
    }
    readonly property real arrowTipY: root.showBelow ? root.targetBounds.y + root.targetBounds.height : root.targetBounds.y

    // Debug: show target bounds
    Rectangle {
        visible: root.debug && root.visible
        x: root.targetBounds.x
        y: root.targetBounds.y
        width: root.targetBounds.width
        height: root.targetBounds.height
        color: "transparent"
        border.color: "red"
        border.width: 2
    }

    // Debug: show arrow tip target point
    Rectangle {
        visible: root.debug && root.visible
        x: root.arrowTipX - 4
        y: root.arrowTipY - 4
        width: 8
        height: 8
        radius: 4
        color: "lime"
    }

    // Debug info text
    Text {
        visible: root.debug && root.visible
        x: 10
        y: 10
        color: "yellow"
        font.pixelSize: 11
        text: "target: " + (root.hint ? root.hint.targetObjectName : "null") +
              "\nfound: " + (root.targetItem ? root.targetItem.objectName : "null") +
              " (" + (root.targetItem ? root.targetItem.width.toFixed(0) + "x" + root.targetItem.height.toFixed(0) : "n/a") + ")" +
              "\nbounds: (" + root.targetBounds.x.toFixed(0) + "," + root.targetBounds.y.toFixed(0) +
              ") " + root.targetBounds.width.toFixed(0) + "x" + root.targetBounds.height.toFixed(0) +
              "\narrowTip: (" + root.arrowTipX.toFixed(0) + "," + root.arrowTipY.toFixed(0) + ")" +
              "\nshowBelow: " + root.showBelow +
              "\nretries: " + retrySearchTimer.attempts
    }

    DropShadow {
        visible: root.showHighlight
        anchors.fill: highlightRect
        source: highlightRect
        horizontalOffset: 0
        verticalOffset: 0
        radius: root.highlightGlowRadius
        samples: 25
        color: Theme.accent
        transparentBorder: true
        z: 0
    }

    Rectangle {
        id: highlightRect
        visible: root.showHighlight
        x: root.targetBounds.x - root.highlightPadding
        y: root.targetBounds.y - root.highlightPadding
        width: root.targetBounds.width + root.highlightPadding * 2
        height: root.targetBounds.height + root.highlightPadding * 2
        radius: root.highlightRadius
        color: "transparent"
        border.color: Theme.accent
        border.width: root.highlightBorderWidth
        z: 1
    }

    // Arrow pointing at target
    Canvas {
        id: arrow
        visible: root.visible
        width: root.arrowSize * 2
        height: root.arrowSize
        x: Math.max(tooltip.x + 12, Math.min(root.arrowTipX - width / 2, tooltip.x + tooltip.width - width - 12))
        y: root.showBelow ? tooltip.y - height + 1 : tooltip.y + tooltip.height - 1
        rotation: root.showBelow ? 0 : 180
        transformOrigin: Item.Center

        onPaint: {
            var ctx = getContext("2d")
            ctx.reset()
            ctx.fillStyle = Theme.accent
            ctx.strokeStyle = Theme.accent
            ctx.lineWidth = 1
            ctx.beginPath()
            ctx.moveTo(0, height)
            ctx.lineTo(width / 2, 0)
            ctx.lineTo(width, height)
            ctx.closePath()
            ctx.fill()
            ctx.stroke()
        }

        Connections {
            target: Theme
            function onAccentChanged() { arrow.requestPaint() }
        }
    }

    // The actual tooltip rectangle
    Rectangle {
        id: tooltip
        visible: root.visible
        width: row.implicitWidth + 28
        height: row.implicitHeight + 16
        color: Theme.accent
        border.color: Theme.accent
        border.width: 1
        radius: Theme.panelRadius

        x: {
            if (!root.targetItem || !root.parent) return 0
            var idealX = root.arrowTipX - width / 2
            return Math.max(8, Math.min(idealX, root.parent.width - width - 8))
        }
        y: {
            if (!root.targetItem || !root.parent) return 0
            if (root.showBelow) {
                return root.arrowTipY + root.gap + root.arrowSize
            } else {
                return root.arrowTipY - height - root.gap - root.arrowSize
            }
        }

        // Block mouse events from passing through to items underneath
        MouseArea {
            anchors.fill: parent
            hoverEnabled: true
            // Accept all mouse buttons to block clicks
            acceptedButtons: Qt.AllButtons
        }

        RowLayout {
            id: row
            anchors.centerIn: parent
            spacing: 8

            Item {
                Layout.preferredWidth: 16
                Layout.preferredHeight: 16
                Layout.alignment: Qt.AlignVCenter

                Image {
                    id: infoIcon
                    anchors.fill: parent
                    source: Theme.iconUrl("info-circle")
                    sourceSize.width: 16
                    sourceSize.height: 16
                    visible: false
                }

                ColorOverlay {
                    anchors.fill: infoIcon
                    source: infoIcon
                    color: Theme.textOnAccent
                }
            }

            Text {
                text: root.hint ? root.hint.message : ""
                color: Theme.textOnAccent
                font.pixelSize: 13
            }

            Text {
                text: "\u00d7"
                color: Theme.textOnAccent
                font.pixelSize: 16

                MouseArea {
                    anchors.fill: parent
                    anchors.margins: -4
                    cursorShape: Qt.PointingHandCursor
                    onClicked: {
                        if (root.hintManager) root.hintManager.dismissCurrentHint()
                    }
                }
            }
        }
    }
}
