pragma ComponentBehavior: Bound

import QtQuick 2.15

/*
  AppShortcuts: Centralizes all keyboard shortcuts.

  Usage in AppRoot:
    AppShortcuts {
        appWorkspaceController: window.appWorkspaceController
        settingsModel: window.settingsModel
        activeFocusItem: window.activeFocusItem
        onSaveProjectAsRequested: saveProjectDialog.openForSave("")
    }
*/
Item {
    id: root

    required property var appWorkspaceController
    required property var settingsModel
    property var activeFocusItem: null

    signal saveProjectAsRequested()

    // Toggle playback
    Shortcut {
        id: togglePlaybackShortcut
        context: Qt.ApplicationShortcut
        property var seqs: root.settingsModel ? root.settingsModel.shortcuts("toggle_playback") : ["Space"]
        sequences: seqs
        enabled: {
            const focusItem = root.activeFocusItem
            return !(focusItem && ("cursorPosition" in focusItem))
        }
        onActivated: {
            if (root.appWorkspaceController) {
                root.appWorkspaceController.togglePlayback()
            }
        }
    }

    // Transcript font size shortcuts
    Shortcut {
        context: Qt.ApplicationShortcut
        sequences: ["Ctrl++", "Ctrl+="]
        onActivated: {
            if (root.settingsModel) {
                var currentSize = root.settingsModel.transcript_font_size()
                root.settingsModel.set_transcript_font_size(currentSize + 2)
            }
        }
    }
    Shortcut {
        context: Qt.ApplicationShortcut
        sequences: ["Ctrl+-"]
        onActivated: {
            if (root.settingsModel) {
                var currentSize = root.settingsModel.transcript_font_size()
                root.settingsModel.set_transcript_font_size(currentSize - 2)
            }
        }
    }

    // Seek shortcuts
    Shortcut {
        id: seekSmallBackShortcut
        context: Qt.ApplicationShortcut
        property var seqs: root.settingsModel ? root.settingsModel.shortcuts("seek_small_back") : [","]
        sequences: seqs
        onActivated: { if (root.appWorkspaceController) root.appWorkspaceController.seekRelative("small", -1) }
    }
    Shortcut {
        id: seekSmallForwardShortcut
        context: Qt.ApplicationShortcut
        property var seqs: root.settingsModel ? root.settingsModel.shortcuts("seek_small_forward") : ["."]
        sequences: seqs
        onActivated: { if (root.appWorkspaceController) root.appWorkspaceController.seekRelative("small", 1) }
    }
    Shortcut {
        id: seekMediumBackShortcut
        context: Qt.ApplicationShortcut
        property var seqs: root.settingsModel ? root.settingsModel.shortcuts("seek_medium_back") : ["J"]
        sequences: seqs
        onActivated: { if (root.appWorkspaceController) root.appWorkspaceController.seekRelative("medium", -1) }
    }
    Shortcut {
        id: seekMediumForwardShortcut
        context: Qt.ApplicationShortcut
        property var seqs: root.settingsModel ? root.settingsModel.shortcuts("seek_medium_forward") : ["L"]
        sequences: seqs
        onActivated: { if (root.appWorkspaceController) root.appWorkspaceController.seekRelative("medium", 1) }
    }
    Shortcut {
        id: seekLargeBackShortcut
        context: Qt.ApplicationShortcut
        property var seqs: root.settingsModel ? root.settingsModel.shortcuts("seek_large_back") : ["Shift+J"]
        sequences: seqs
        onActivated: { if (root.appWorkspaceController) root.appWorkspaceController.seekRelative("large", -1) }
    }
    Shortcut {
        id: seekLargeForwardShortcut
        context: Qt.ApplicationShortcut
        property var seqs: root.settingsModel ? root.settingsModel.shortcuts("seek_large_forward") : ["Shift+L"]
        sequences: seqs
        onActivated: { if (root.appWorkspaceController) root.appWorkspaceController.seekRelative("large", 1) }
    }

    // Speed up/down shortcuts
    Shortcut {
        id: speedUpShortcut
        context: Qt.ApplicationShortcut
        property var seqs: root.settingsModel ? root.settingsModel.shortcuts("speed_up") : ["]"]
        sequences: seqs
        onActivated: {
            if (!root.appWorkspaceController || !root.settingsModel) return
            var speeds = root.settingsModel.playback_speeds()
            var current = root.appWorkspaceController.playbackSpeed
            for (var i = 0; i < speeds.length; i++) {
                if (speeds[i] > current + 0.001) {
                    root.appWorkspaceController.setPlaybackSpeed(speeds[i])
                    return
                }
            }
        }
    }
    Shortcut {
        id: speedDownShortcut
        context: Qt.ApplicationShortcut
        property var seqs: root.settingsModel ? root.settingsModel.shortcuts("speed_down") : ["["]
        sequences: seqs
        onActivated: {
            if (!root.appWorkspaceController || !root.settingsModel) return
            var speeds = root.settingsModel.playback_speeds()
            var current = root.appWorkspaceController.playbackSpeed
            for (var i = speeds.length - 1; i >= 0; i--) {
                if (speeds[i] < current - 0.001) {
                    root.appWorkspaceController.setPlaybackSpeed(speeds[i])
                    return
                }
            }
        }
    }

    // Cut/uncut selection shortcuts
    Shortcut {
        id: cutSelectionShortcut
        context: Qt.ApplicationShortcut
        property var seqs: root.settingsModel ? root.settingsModel.shortcuts("cut_selection") : ["X", "Backspace", "Del"]
        sequences: seqs
        onActivated: {
            if (root.appWorkspaceController && root.appWorkspaceController.cutWorkspaceController) {
                root.appWorkspaceController.cutWorkspaceController.cutSelection()
            }
        }
    }
    Shortcut {
        id: cutUnselectedShortcut
        context: Qt.ApplicationShortcut
        property var seqs: root.settingsModel ? root.settingsModel.shortcuts("cut_unselected") : ["Shift+Backspace", "Shift+Del"]
        sequences: seqs
        onActivated: {
            if (root.appWorkspaceController && root.appWorkspaceController.cutWorkspaceController) {
                root.appWorkspaceController.cutWorkspaceController.cutUnselected()
            }
        }
    }
    Shortcut {
        id: uncutSelectionShortcut
        context: Qt.ApplicationShortcut
        property var seqs: root.settingsModel ? root.settingsModel.shortcuts("uncut_selection") : ["Shift+X"]
        sequences: seqs
        onActivated: {
            if (root.appWorkspaceController && root.appWorkspaceController.cutWorkspaceController) {
                root.appWorkspaceController.cutWorkspaceController.uncutSelection()
            }
        }
    }

    // Undo/redo shortcuts
    Shortcut {
        id: undoShortcut
        context: Qt.ApplicationShortcut
        property var seqs: root.settingsModel ? root.settingsModel.shortcuts("undo") : ["Ctrl+Z"]
        sequences: seqs
        onActivated: {
            if (root.appWorkspaceController) {
                root.appWorkspaceController.undo()
            }
        }
    }
    Shortcut {
        id: redoShortcut
        context: Qt.ApplicationShortcut
        property var seqs: root.settingsModel ? root.settingsModel.shortcuts("redo") : ["Ctrl+Shift+Z", "Ctrl+Y"]
        sequences: seqs
        onActivated: {
            if (root.appWorkspaceController) {
                root.appWorkspaceController.redo()
            }
        }
    }

    // Mark in/out shortcuts
    Shortcut {
        id: markInShortcut
        context: Qt.ApplicationShortcut
        property var seqs: root.settingsModel ? root.settingsModel.shortcuts("mark_in") : ["I"]
        sequences: seqs
        onActivated: {
            if (root.appWorkspaceController) {
                root.appWorkspaceController.markIn()
            }
        }
    }
    Shortcut {
        id: markOutShortcut
        context: Qt.ApplicationShortcut
        property var seqs: root.settingsModel ? root.settingsModel.shortcuts("mark_out") : ["O"]
        sequences: seqs
        onActivated: {
            if (root.appWorkspaceController) {
                root.appWorkspaceController.markOut()
            }
        }
    }
    Shortcut {
        id: clearSelectionShortcut
        context: Qt.ApplicationShortcut
        property var seqs: root.settingsModel ? root.settingsModel.shortcuts("clear_selection") : ["Escape"]
        sequences: seqs
        onActivated: {
            if (root.appWorkspaceController) {
                root.appWorkspaceController.clearSelection()
            }
        }
    }
    Shortcut {
        id: selectAllShortcut
        context: Qt.ApplicationShortcut
        property var seqs: root.settingsModel ? root.settingsModel.shortcuts("select_all") : ["Ctrl+A"]
        sequences: seqs
        onActivated: {
            if (root.appWorkspaceController) {
                root.appWorkspaceController.selectAll()
            }
        }
    }
    Shortcut {
        id: selectToStartShortcut
        context: Qt.ApplicationShortcut
        property var seqs: root.settingsModel ? root.settingsModel.shortcuts("select_to_start") : ["Shift+Home"]
        sequences: seqs
        onActivated: {
            if (root.appWorkspaceController) {
                root.appWorkspaceController.selectToStart()
            }
        }
    }
    Shortcut {
        id: selectToEndShortcut
        context: Qt.ApplicationShortcut
        property var seqs: root.settingsModel ? root.settingsModel.shortcuts("select_to_end") : ["Shift+End"]
        sequences: seqs
        onActivated: {
            if (root.appWorkspaceController) {
                root.appWorkspaceController.selectToEnd()
            }
        }
    }

    // Loop shortcuts
    Shortcut {
        id: loopSelectionShortcut
        context: Qt.ApplicationShortcut
        property var seqs: root.settingsModel ? root.settingsModel.shortcuts("loop_selection") : ["G"]
        sequences: seqs
        onActivated: {
            if (!root.appWorkspaceController) return
            var backend = root.appWorkspaceController.timelineBackend
            if (!backend) return
            var selRange = backend.selectionRange
            if (selRange && selRange.length === 2) {
                var a = Math.min(selRange[0], selRange[1])
                var b = Math.max(selRange[0], selRange[1])
                if (Math.abs(b - a) > 0.001) {
                    root.appWorkspaceController.loopSelectionRange(a, b)
                }
            }
        }
    }
    Shortcut {
        id: clearLoopShortcut
        context: Qt.ApplicationShortcut
        property var seqs: root.settingsModel ? root.settingsModel.shortcuts("clear_loop") : ["Shift+G"]
        sequences: seqs
        onActivated: {
            if (root.appWorkspaceController) {
                root.appWorkspaceController.clearLoop()
            }
        }
    }

    Shortcut {
        id: centerSelectionShortcut
        context: Qt.ApplicationShortcut
        property var seqs: root.settingsModel ? root.settingsModel.shortcuts("center_selection") : ["C"]
        sequences: seqs
        onActivated: {
            if (!root.appWorkspaceController) return
            var backend = root.appWorkspaceController.timelineBackend
            if (backend) backend.centerOnSelection()
        }
    }

    Shortcut {
        id: saveProjectShortcut
        context: Qt.ApplicationShortcut
        property var seqs: root.settingsModel ? root.settingsModel.shortcuts("save_project") : ["Ctrl+S"]
        sequences: seqs
        onActivated: {
            if (root.appWorkspaceController) {
                root.appWorkspaceController.saveProject()
            }
        }
    }

    Shortcut {
        id: saveProjectAsShortcut
        context: Qt.ApplicationShortcut
        property var seqs: root.settingsModel ? root.settingsModel.shortcuts("save_project_as") : ["Ctrl+Shift+S"]
        sequences: seqs
        onActivated: root.saveProjectAsRequested()
    }

    // Zoom shortcuts
    Shortcut {
        id: zoomInShortcut
        context: Qt.ApplicationShortcut
        property var seqs: root.settingsModel ? root.settingsModel.shortcuts("zoom_in") : ["=", "+"]
        sequences: seqs
        onActivated: {
            if (root.appWorkspaceController && root.appWorkspaceController.timelineBackend) {
                root.appWorkspaceController.timelineBackend.zoomIn()
            }
        }
    }
    Shortcut {
        id: zoomOutShortcut
        context: Qt.ApplicationShortcut
        property var seqs: root.settingsModel ? root.settingsModel.shortcuts("zoom_out") : ["-"]
        sequences: seqs
        onActivated: {
            if (root.appWorkspaceController && root.appWorkspaceController.timelineBackend) {
                root.appWorkspaceController.timelineBackend.zoomOut()
            }
        }
    }

    // Update shortcuts when settings change
    Connections {
        target: root.settingsModel || null
        function onShortcut_changed(name, sequences) {
            switch (name) {
            case "toggle_playback":
                togglePlaybackShortcut.seqs = sequences
                break
            case "seek_small_back":
                seekSmallBackShortcut.seqs = sequences
                break
            case "seek_small_forward":
                seekSmallForwardShortcut.seqs = sequences
                break
            case "seek_medium_back":
                seekMediumBackShortcut.seqs = sequences
                break
            case "seek_medium_forward":
                seekMediumForwardShortcut.seqs = sequences
                break
            case "seek_large_back":
                seekLargeBackShortcut.seqs = sequences
                break
            case "seek_large_forward":
                seekLargeForwardShortcut.seqs = sequences
                break
            case "speed_up":
                speedUpShortcut.seqs = sequences
                break
            case "speed_down":
                speedDownShortcut.seqs = sequences
                break
            case "cut_selection":
                cutSelectionShortcut.seqs = sequences
                break
            case "cut_unselected":
                cutUnselectedShortcut.seqs = sequences
                break
            case "uncut_selection":
                uncutSelectionShortcut.seqs = sequences
                break
            case "undo":
                undoShortcut.seqs = sequences
                break
            case "redo":
                redoShortcut.seqs = sequences
                break
            case "mark_in":
                markInShortcut.seqs = sequences
                break
            case "mark_out":
                markOutShortcut.seqs = sequences
                break
            case "clear_selection":
                clearSelectionShortcut.seqs = sequences
                break
            case "select_all":
                selectAllShortcut.seqs = sequences
                break
            case "select_to_start":
                selectToStartShortcut.seqs = sequences
                break
            case "select_to_end":
                selectToEndShortcut.seqs = sequences
                break
            case "loop_selection":
                loopSelectionShortcut.seqs = sequences
                break
            case "clear_loop":
                clearLoopShortcut.seqs = sequences
                break
            case "center_selection":
                centerSelectionShortcut.seqs = sequences
                break
            case "save_project":
                saveProjectShortcut.seqs = sequences
                break
            case "save_project_as":
                saveProjectAsShortcut.seqs = sequences
                break
            case "zoom_in":
                zoomInShortcut.seqs = sequences
                break
            case "zoom_out":
                zoomOutShortcut.seqs = sequences
                break
            }
        }
    }
}
